// Gradient Background
// Fills pillarbox/letterbox areas with a customizable gradient

//!BGFX EFFECT
//!VERSION 1
//!NAME Gradient
//!CATEGORY Background
//!DESCRIPTION Fills the black bars with a smooth gradient between two colors.

//!PARAMETER
//!LABEL Color 1 Red
//!DEFAULT 0.05
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color1R;

//!PARAMETER
//!LABEL Color 1 Green
//!DEFAULT 0.0
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color1G;

//!PARAMETER
//!LABEL Color 1 Blue
//!DEFAULT 0.15
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color1B;

//!PARAMETER
//!LABEL Color 2 Red
//!DEFAULT 0.15
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color2R;

//!PARAMETER
//!LABEL Color 2 Green
//!DEFAULT 0.0
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color2G;

//!PARAMETER
//!LABEL Color 2 Blue
//!DEFAULT 0.05
//!MIN 0.0
//!MAX 1.0
//!STEP 0.01
float color2B;

//!PARAMETER
//!LABEL Gradient Angle
//!DESC Direction of the gradient in degrees (0 = horizontal, 90 = vertical)
//!DEFAULT 45
//!MIN 0
//!MAX 360
//!STEP 15
float gradientAngle;

//!TEXTURE
Texture2D INPUT;

//!TEXTURE
//!WIDTH RENDERER_WIDTH
//!HEIGHT RENDERER_HEIGHT
Texture2D OUTPUT;

//!SAMPLER
//!FILTER LINEAR
SamplerState linearSampler;

//!PASS 1
//!DESC Gradient Background
//!STYLE PS
//!IN INPUT
//!OUT OUTPUT

float4 Pass1(float2 pos) {
    float2 inputSize = float2(GetInputSize());
    float2 outputSize = float2(GetOutputSize());

    float inputAspect = inputSize.x / inputSize.y;
    float outputAspect = outputSize.x / outputSize.y;

    float2 contentScale;
    float2 contentOffset;

    if (inputAspect > outputAspect) {
        contentScale = float2(1.0, outputAspect / inputAspect);
        contentOffset = float2(0.0, (1.0 - contentScale.y) * 0.5);
    } else {
        contentScale = float2(inputAspect / outputAspect, 1.0);
        contentOffset = float2((1.0 - contentScale.x) * 0.5, 0.0);
    }

    float2 contentUV = (pos - contentOffset) / contentScale;
    bool insideContent = all(contentUV >= 0.0) && all(contentUV <= 1.0);

    if (insideContent) {
        return INPUT.SampleLevel(linearSampler, contentUV, 0);
    }

    // Calculate gradient
    float3 color1 = float3(color1R, color1G, color1B);
    float3 color2 = float3(color2R, color2G, color2B);

    float angleRad = radians(gradientAngle);
    float2 dir = float2(cos(angleRad), sin(angleRad));
    float t = dot(pos - 0.5, dir) + 0.5;
    t = saturate(t);

    float3 color = lerp(color1, color2, t);

    return float4(color, 1.0);
}
