// RAVU 3x Upscaling Shader (R2 Variant, Luma)
// Rapid and Accurate Video Upscaling - Triple scale version
// Edge-adaptive upscaler using radius 2 kernel for 3x scaling

//!BGFX EFFECT
//!VERSION 1
//!NAME RAVU 3x R2
//!CATEGORY Upscaling
//!DESCRIPTION 3x edge-adaptive upscaler with radius 2 kernel. Processes luminance channel only. Fast but lower quality.

//!TEXTURE
Texture2D INPUT;

//!SAMPLER
//!FILTER POINT
SamplerState sam_INPUT;

//!TEXTURE
//!WIDTH  INPUT_WIDTH * 3
//!HEIGHT INPUT_HEIGHT * 3
Texture2D OUTPUT;

//!SAMPLER
//!FILTER LINEAR
SamplerState sam_INPUT_LINEAR;

//!TEXTURE
//!SOURCE ravu_3x_lut2_f16.dds
//!FORMAT R16G16B16A16_FLOAT
Texture2D ravu_3x_lut2;

//!SAMPLER
//!FILTER LINEAR
SamplerState sam_ravu_3x_lut2;

//!COMMON
#include "prescalers.hlsli"

#define LAST_PASS 1

//!PASS 1
//!DESC RAVU-3x Upscale (luma, r2)
//!IN INPUT, ravu_3x_lut2
//!OUT OUTPUT
//!BLOCK_SIZE 96, 24
//!NUM_THREADS 32, 8
shared float inp[340];

#define CURRENT_PASS 1

#define GET_SAMPLE(x) dot(x.rgb, rgb2y)
#define imageStore(out_image, pos, val) imageStoreOverride(pos, val.x)
void imageStoreOverride(uint2 pos, float value) {
	float2 UV = mul(rgb2uv, INPUT.SampleLevel(sam_INPUT_LINEAR, HOOKED_map(pos), 0).rgb);
	OUTPUT[pos] = float4(mul(yuv2rgb, float3(value.x, UV)), 1.0);
}

#define INPUT_tex(pos) GET_SAMPLE(vec4(texture(INPUT, pos)))
static const float2 INPUT_size = float2(GetInputSize());
static const float2 INPUT_pt = float2(GetInputPt());

#define ravu_3x_lut2_tex(pos) (vec4(texture(ravu_3x_lut2, pos)))

#define HOOKED_tex(pos) INPUT_tex(pos)
#define HOOKED_size INPUT_size
#define HOOKED_pt INPUT_pt

void Pass1(uint2 blockStart, uint3 threadId) {
	ivec2 group_base = ivec2(gl_WorkGroupID) * ivec2(gl_WorkGroupSize);
	int local_pos = int(gl_LocalInvocationID.x) * 10 + int(gl_LocalInvocationID.y);
	for (int id = int(gl_LocalInvocationIndex); id < 340; id += int(gl_WorkGroupSize.x * gl_WorkGroupSize.y)) {
		uint x = (uint)id / 10, y = (uint)id % 10;
		inp[id] = HOOKED_tex(HOOKED_pt * vec2(float(group_base.x + x) + (-0.5), float(group_base.y + y) + (-0.5))).x;
	}
	barrier();
#if CURRENT_PASS == LAST_PASS
	uint2 destPos = blockStart + threadId.xy * 3;
	uint2 outputSize = GetOutputSize();
	if (destPos.x >= outputSize.x || destPos.y >= outputSize.y) {
		return;
	}
#endif

	// Sample luminance values from shared memory
	float luma0 = inp[local_pos + 0];
	float luma1 = inp[local_pos + 1];
	float luma2 = inp[local_pos + 2];
	float luma3 = inp[local_pos + 10];
	float luma4 = inp[local_pos + 11];
	float luma5 = inp[local_pos + 12];
	float luma6 = inp[local_pos + 20];
	float luma7 = inp[local_pos + 21];
	float luma8 = inp[local_pos + 22];

	// Structure tensor computation for edge detection
	vec3 abd = vec3(0.0, 0.0, 0.0);
	float gx, gy;
	gx = (luma3 - luma0);
	gy = (luma1 - luma0);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.1018680644198163;
	gx = (luma4 - luma1);
	gy = (luma2 - luma0) / 2.0;
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.11543163961422666;
	gx = (luma5 - luma2);
	gy = (luma2 - luma1);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.1018680644198163;
	gx = (luma6 - luma0) / 2.0;
	gy = (luma4 - luma3);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.11543163961422666;
	gx = (luma7 - luma1) / 2.0;
	gy = (luma5 - luma3) / 2.0;
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.13080118386382833;
	gx = (luma8 - luma2) / 2.0;
	gy = (luma5 - luma4);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.11543163961422666;
	gx = (luma6 - luma3);
	gy = (luma7 - luma6);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.1018680644198163;
	gx = (luma7 - luma4);
	gy = (luma8 - luma6) / 2.0;
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.11543163961422666;
	gx = (luma8 - luma5);
	gy = (luma8 - luma7);
	abd += vec3(gx * gx, gx * gy, gy * gy) * 0.1018680644198163;

	// Eigenvalue decomposition
	float a = abd.x, b = abd.y, d = abd.z;
	float T = a + d, D = a * d - b * b;
	float delta = sqrt(max(T * T / 4.0 - D, 0.0));
	float L1 = T / 2.0 + delta, L2 = T / 2.0 - delta;
	float sqrtL1 = sqrt(L1), sqrtL2 = sqrt(L2);
	float theta = mix(mod(atan(L1 - a, b) + 3.141592653589793, 3.141592653589793), 0.0, abs(b) < 1.192092896e-7);
	float lambda = sqrtL1;
	float mu = mix((sqrtL1 - sqrtL2) / (sqrtL1 + sqrtL2), 0.0, sqrtL1 + sqrtL2 < 1.192092896e-7);

	// LUT coordinate calculation
	float angle = floor(theta * 24.0 / 3.141592653589793);
	float strength = mix(mix(0.0, 1.0, lambda >= 0.005), 2.0, lambda >= 0.02);
	float coherence = mix(mix(0.0, 1.0, mu >= 0.25), 2.0, mu >= 0.5);
	float coord_y = ((angle * 3.0 + strength) * 3.0 + coherence + 0.5) / 216.0;

	// Weighted sample accumulation from LUT
	vec4 res0 = vec4(0.0, 0.0, 0.0, 0.0), res1 = vec4(0.0, 0.0, 0.0, 0.0);
	vec4 w0, w1;
	w0 = texture(ravu_3x_lut2, vec2(0.05, coord_y));
	w1 = texture(ravu_3x_lut2, vec2(0.15, coord_y));
	res0 += luma0 * w0 + luma8 * w1.wzyx;
	res1 += luma0 * w1 + luma8 * w0.wzyx;
	w0 = texture(ravu_3x_lut2, vec2(0.25, coord_y));
	w1 = texture(ravu_3x_lut2, vec2(0.35, coord_y));
	res0 += luma1 * w0 + luma7 * w1.wzyx;
	res1 += luma1 * w1 + luma7 * w0.wzyx;
	w0 = texture(ravu_3x_lut2, vec2(0.45, coord_y));
	w1 = texture(ravu_3x_lut2, vec2(0.55, coord_y));
	res0 += luma2 * w0 + luma6 * w1.wzyx;
	res1 += luma2 * w1 + luma6 * w0.wzyx;
	w0 = texture(ravu_3x_lut2, vec2(0.65, coord_y));
	w1 = texture(ravu_3x_lut2, vec2(0.75, coord_y));
	res0 += luma3 * w0 + luma5 * w1.wzyx;
	res1 += luma3 * w1 + luma5 * w0.wzyx;
	w0 = texture(ravu_3x_lut2, vec2(0.85, coord_y));
	w1 = texture(ravu_3x_lut2, vec2(0.95, coord_y));
	res0 += luma4 * w0;
	res1 += luma4 * w1;
	res0 = clamp(res0, 0.0, 1.0);
	res1 = clamp(res1, 0.0, 1.0);

	// Output 3x3 block
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(0, 0), res0[0]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(0, 1), res0[1]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(0, 2), res0[2]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(1, 0), res0[3]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(1, 1), luma4);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(1, 2), res1[0]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(2, 0), res1[1]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(2, 1), res1[2]);
	imageStore(out_image, ivec2(gl_GlobalInvocationID) * 3 + ivec2(2, 2), res1[3]);
}
