const int BIOME_BITMAP_X_SIZE = 6;
const int BIOME_BITMAP_Z_SIZE = 6;
const int BIOME_BITMAP_TOTAL_SIZE = BIOME_BITMAP_X_SIZE * BIOME_BITMAP_Z_SIZE;
const int BIOME_LIGHTING_DAY_DIVISIONS = 7;
const int BILLBOARD_VERTICES = 6;
const int MAX_BIOMES = 16;
const int MAX_BIOME_PLANTS = 16;

uniform float dayPercent;
uniform vec3 eyePoint;
uniform ivec3 cameraOffset;
uniform ivec3 worldOffset;

#include "common/constants.glsl"

struct BitmapPosition {
	vec2 start;
	vec2 size;
	int layer;
	int unused;
	vec2 vertexPositions[BILLBOARD_VERTICES];
};

struct Plant {
	float height;
	float width;
	float trunkColourRed, trunkColourGreen, trunkColourBlue;
	float renderBillboards;
	float floatsOnWater;
	float maxWaterDepth;
	BitmapPosition rotations[BIOME_BITMAP_TOTAL_SIZE];
};

layout(std430, binding = 0) buffer PlantDefinitions
{
    Plant plantDefinitions[];
};

struct PlantInstance {
	int id;
	float probabilityLimit;
};

struct LightingInfo {
	float treeLitPercent;
	float diffuseForLit;
	float underlayLitPercent;
	float unused;
};

struct LightingHour {
	vec4 lightingDirections[4];
};

struct Biome {
	PlantInstance trees[MAX_BIOME_PLANTS];
	PlantInstance shrubs[MAX_BIOME_PLANTS];
	LightingHour lightingHours[BIOME_LIGHTING_DAY_DIVISIONS * 2];
	float normalTreeCoverage, winterTreeCoverage;
	float treeUnderlayNoiseDivisor, treeLitNoiseDivisor;
	// If this is changed to add more than 0, 1 as valid values
	// then the land fragment shader will need to be changed to support it
	// as this integer is used as an index into a vector that only has as many elements as needed
	int underlayIndex, unused1, unused2, unused3;
};

layout(std430, binding = 1) buffer BiomeDefinitions
{
	Biome biomeDefinitions[MAX_BIOMES];
};

struct BiomeLightingComponents {
	int timeOfDayFloor, timeOfDayCeil;
	int directionFloor, directionCeil;
	float timeOfDayFract, directionFract;
	bool snow;
};

BiomeLightingComponents getLightingComponents(vec3 worldPosition, bool snow) {
	BiomeLightingComponents lighting;

	ivec3 offset = worldOffset - cameraOffset;
	float eyeRelativeAngle = clamp((atan((eyePoint.x + offset.x) - worldPosition.x, worldPosition.z - (eyePoint.z + offset.z)) + M_PI) / (M_PI / 2.f), 0.f, 4.f);
	lighting.directionFloor = int(floor(eyeRelativeAngle));
	lighting.directionCeil = int(ceil(eyeRelativeAngle)) % 4;
	lighting.directionFract = fract(eyeRelativeAngle);

	const int dayDivisions = (BIOME_LIGHTING_DAY_DIVISIONS - 1);
	lighting.timeOfDayFloor = clamp(int(floor(dayPercent * dayDivisions)), 0, dayDivisions) + (snow ? 7 : 0);
	lighting.timeOfDayCeil = clamp(int(ceil(dayPercent * dayDivisions)), 0, dayDivisions) + (snow ? 7 : 0);
	lighting.timeOfDayFract = fract(dayPercent * dayDivisions);

	lighting.snow = snow;

	return lighting;
}

vec4 getBiomeLighting(int biomeId, vec3 worldPosition, bool snow) {
	ivec3 offset = worldOffset - cameraOffset;
	float eyeRelativeAngle = clamp((atan((eyePoint.x + offset.x) - worldPosition.x, worldPosition.z - (eyePoint.z + offset.z)) + M_PI) / (M_PI / 2.f), 0.f, 4.f);
	int directionFloor = int(floor(eyeRelativeAngle)), directionCeil = int(ceil(eyeRelativeAngle)) % 4;
	float directionFract = fract(eyeRelativeAngle);

	const int dayDivisions = (BIOME_LIGHTING_DAY_DIVISIONS - 1);
	int lightingTimeOfDayFloor = clamp(int(floor(dayPercent * dayDivisions)), 0, dayDivisions) + (snow ? 7 : 0);
	int lightingTimeOfDayCeil = clamp(int(ceil(dayPercent * dayDivisions)), 0, dayDivisions) + (snow ? 7 : 0);
	float lightingTimeOfDayFract = fract(dayPercent * dayDivisions);

	vec4 biome1LightingHourFloor = 
		(biomeDefinitions[biomeId].lightingHours[lightingTimeOfDayFloor].lightingDirections[directionFloor] * (1.f - directionFract)) +
		(biomeDefinitions[biomeId].lightingHours[lightingTimeOfDayFloor].lightingDirections[directionCeil] * directionFract);

	vec4 biome1LightingHourCeil = 
		(biomeDefinitions[biomeId].lightingHours[lightingTimeOfDayCeil].lightingDirections[directionFloor] * (1.f - directionFract)) +
		(biomeDefinitions[biomeId].lightingHours[lightingTimeOfDayCeil].lightingDirections[directionCeil] * directionFract);

	return (biome1LightingHourFloor * (1.f - lightingTimeOfDayFract)) + (biome1LightingHourCeil * lightingTimeOfDayFract);
}

vec4 getBiomeLighting2(int biomeId, const BiomeLightingComponents lighting) {
	vec4 biome1LightingHourFloor = 
		(biomeDefinitions[biomeId].lightingHours[lighting.timeOfDayFloor].lightingDirections[lighting.directionFloor] * (1.f - lighting.directionFract)) +
		(biomeDefinitions[biomeId].lightingHours[lighting.timeOfDayFloor].lightingDirections[lighting.directionCeil] * lighting.directionFract);

	vec4 biome1LightingHourCeil = 
		(biomeDefinitions[biomeId].lightingHours[lighting.timeOfDayCeil].lightingDirections[lighting.directionFloor] * (1.f - lighting.directionFract)) +
		(biomeDefinitions[biomeId].lightingHours[lighting.timeOfDayCeil].lightingDirections[lighting.directionCeil] * lighting.directionFract);

	return (biome1LightingHourFloor * (1.f - lighting.timeOfDayFract)) + (biome1LightingHourCeil * lighting.timeOfDayFract);
}

const int MAX_CURRENCIES = 3;

/* For a given currency, how much each biome would generate, normalised between 0.0 - 1.0 */
struct CurrencyBiomeMapping {
	float biomeIntensity[MAX_BIOMES];
};

layout(std430, binding = 2) buffer CurrencyBiomeMappings
{
	CurrencyBiomeMapping currencyBiomeMappings[MAX_CURRENCIES];
};

const int MAX_EXISTING_BUILDINGS = 8;

struct CurrencyBuilding {
	vec2 worldPosition;
	float range;
	float valid;
};

layout(std430, binding = 3) buffer CurrencyBuildings
{
	CurrencyBuilding currencyBuildings[MAX_EXISTING_BUILDINGS];
};
