/* Data needed for large sediment to be deposited at a position */
struct SedimentDepositionInfo {
	/* The height per tick that should be deposited */
	float increase;
	/* The maximum height that land should rise to as a result of this sediment depositing */
	float maximumHeight;
};

/* Packs a SedimentDepositionInfo struct into a single 32-bit floating point number */
float packSedimentDeposition(SedimentDepositionInfo sedimentDepositionInfo) {
	/* The first 2 bits are hardcoded both to 1, so that the final floating point number is valid
	   Otherwise if the step was 0 then exponent of the resulting number could be all zeros and 
	   fail to produce a validly encoded floating point number.  The exponent cannot be all ones
	   either */
	uint firstBits = 0x3u << 30;
	/* The next 13 bits encode the increase as an unsigned int.  Values that can be represented
	   are between 0 and 0.015625 with a resolution of 0.0000019 */
	uint increase = clamp(uint(ceil(sedimentDepositionInfo.increase * 524288.f)), 0u, 0x1fffu) << 17;
	/* The last 17 bits encode the maximum height, storing an unsigned int that should be
	   divided by 2 to retrieve the actual height.  Values that can be represented are
	   between 0 and 4096 with a resolution 0.03125 (1 / 32) */
	uint maximumHeight = clamp(uint(sedimentDepositionInfo.maximumHeight * 32.f), 0u, 0x1ffffu);
	return uintBitsToFloat(firstBits | increase | maximumHeight);
}

/* Unpacks a single floating point number into a SedimentDepositionInfo struct */
SedimentDepositionInfo unpackSedimentDeposition(float sedimentDeposition) {
	uint value = floatBitsToUint(sedimentDeposition);
	float maximumHeight = (value & 0x0001ffffu) / 32.f;
	float increase = ((value & 0x3ffe0000u) >> 17) / 524288.f;
	return SedimentDepositionInfo(increase, maximumHeight);
}

struct WaterData {
	float landHeight, waterHeight, totalHeight, waterDisplayHeight;
	vec4 flows;
	int primaryFlowDirection;
	float groundWater, unused1, rainfall;
	float floodSediment, sediment, floodHeight, connectsOcean;
};

WaterData getWaterData(ivec2 dataPos, bool fetchRow2, bool fetchRow3, bool fetchRow4) {
	vec4 row1 = texelFetch(water0Read, dataPos);
	vec4 row2 = vec4(0);
	if (fetchRow2) {
		row2 = texelFetch(water1Read, dataPos);
	}
	vec4 row3 = vec4(0);
	if (fetchRow3) {
		row3 = texelFetch(water2Read, dataPos);
	}
	vec4 row4 = vec4(0);
	if (fetchRow4) {
		row4 = texelFetch(water3Read, dataPos);
	}

	return WaterData(
		row1[0], row1[1], row1[2], row1[3],
		vec4(row2[0], row2[1], row2[2], row2[3]),
		int(row3[0]), row3[1], row3[2], row3[3],
		row4[0], row4[1], row4[2], row4[3]
	);
}

void outputWaterData(inout WaterData data, inout vec4 row1, inout vec4 row2, inout vec4 row3, inout vec4 row4) {
	row1 = vec4(data.landHeight, data.waterHeight, data.totalHeight, data.waterDisplayHeight);
	row2 = vec4(data.flows[0], data.flows[1], data.flows[2], data.flows[3]);
	row3 = vec4(float(data.primaryFlowDirection), data.groundWater, data.unused1, data.rainfall);
	row4 = vec4(data.floodSediment, data.sediment, data.floodHeight, data.connectsOcean);
}

float calculateTotalHeight(WaterData data, vec4 lavaRow0, vec4 iceRow1) {
	return data.landHeight + max(max(0.f, data.waterDisplayHeight), max(0.f, iceRow1[0])) + max(0.f, lavaRow0[0]);
}