#version 430

#include "common/constants.glsl"
#include "common/data/biomes.glsl"
#include "common/octohedral.glsl"

uniform sampler2D coordinatesTexture;
uniform bool usingTextureAtlas;
uniform mat4 ciModelViewProjectionInverse;
uniform mat4 ciModelViewProjection;
uniform mat4 ciViewMatrix;
uniform mat4 ciViewMatrixInverse;
uniform bool centreOnGround;
uniform float lightDirectionMix;
uniform bool shadowsEnabled;
uniform float SHADOWS_MIX;
uniform float GLOBAL_SCALE;
uniform int INTERPOLATED_VIEWS;
uniform float WIDTH_SCALE;

layout(points) in;
layout(triangle_strip, max_vertices=4) out;

in vec4 vColour[];
in float vSkip[];
in int vType[];
in float vRotation[];
in float vScale[];
in float vDepth[];
in float vOpacity[];
in float vIllumination[];
in vec2 vShadows1[];
in vec2 vShadows2[];

out vec4 fColour;
out vec2 fTexCoord;
out float fOpacity;
out float fDiffuse;
out float fAmbient;
out float fRotation;
out vec3 rotatedWorldPosition;
out vec3 centrePosition;
flat out ivec3 views;
flat out vec3 viewRatios; 
flat out int type;
flat out float fScale;
flat out vec3 cameraEyePosition;

vec3 rotate(vec3 v, vec3 axis, float angle) {
    float cosTheta = cos(angle);
    float sinTheta = sin(angle);

    return (v * cosTheta) +
           (cross(axis, v) * sinTheta) +
           (axis * dot(axis, v) * (1.0 - cosTheta));
                   
}

mat4 makeYRotationMatrix(float theta) {
    float cosTheta = cos(theta);
    float sinTheta = sin(theta);
    
    return mat4(
        cosTheta,  0.0, sinTheta, 0.0,
        0.0,       1.0, 0.0,      0.0,
        -sinTheta, 0.0, cosTheta, 0.0,
        0.0,       0.0, 0.0,      1.0
    );
}

int getOctoView(int x, int y) {
	return x + (y * BIOME_BITMAP_X_SIZE);
}

vec3 additiveNormalise(vec3 value) {
	return value / (dot(value, vec3(1)));
}

void sortViews(inout ivec3 views, inout vec3 viewRatios) {
	int temp;
	float _temp;
	if (viewRatios[0] < viewRatios[1]) {
		_temp = viewRatios[1];
		viewRatios[1] = viewRatios[0];
		viewRatios[0] = _temp;

		temp = views[1];
		views[1] = views[0];
		views[0] = temp;
	}

	if (viewRatios[1] < viewRatios[2]) {
		_temp = viewRatios[2];
		viewRatios[2] = viewRatios[1];
		viewRatios[1] = _temp;

		temp = views[2];
		views[2] = views[1];
		views[1] = temp;
	}

	if (viewRatios[0] < viewRatios[1]) {
		_temp = viewRatios[1];
		viewRatios[1] = viewRatios[0];
		viewRatios[0] = _temp;

		temp = views[1];
		views[1] = views[0];
		views[0] = temp;
	}
}

// const vec2 vertexPositions[6] = vec2[6](
// 	vec2(1.f, -1.f),
// 	vec2(-1.f, -1.f),
// 	vec2(1.5f, 0.f),
// 	vec2(-1.5f, 0.f),
// 	vec2(1.f, 1.f),
// 	vec2(-1.f, 1.f)
// );

const vec2 vertexPositions[4] = vec2[4](
	vec2(-1.f, -1.f),
	vec2(1.f, -1.f),
	vec2(-1.f, 1.f),
	vec2(1.f, 1.f)
);

void main() {	
	vec4 position = gl_in[0].gl_Position;
	type = vType[0];
	fScale = vScale[0] * GLOBAL_SCALE;
	float opacity = vOpacity[0];
	fAmbient = vIllumination[0];

	float worldHeight = position.y - cameraOffset.y;

	if (shadowsEnabled) {
		float _shadows1 = 1.f;
		if (worldHeight< vShadows1[0][0]) {
			float shadowDist = vShadows1[0][1];
			float shadowBottom = vShadows1[0][0] - shadowDist;
			if (worldHeight < shadowBottom) {
				_shadows1 = 0.f;
			} else {
				_shadows1 = mix(1.0, 0.f, (vShadows1[0][0] - worldHeight) / shadowDist);
			}
		}

		float _shadows2 = 1.f;
		if (worldHeight < vShadows2[0][0]) {
			float shadowDist = vShadows2[0][1];
			float shadowBottom = vShadows2[0][0] - shadowDist;
			if (worldHeight < shadowBottom) {
				_shadows2 = 0.f;
			} else {
				_shadows2 = mix(1.0, 0.f, (vShadows2[0][0] - worldHeight) / shadowDist);
			}
		}

		fDiffuse = vIllumination[0] * mix(_shadows1, _shadows2, lightDirectionMix);
	} else {
		fDiffuse = vIllumination[0];
	}

	if (vSkip[0] != 1.0 && opacity > 0.f) {
		fOpacity = opacity;
		fColour = vColour[0];
		fRotation = vRotation[0];

		vec3 cameraRight = vec3(ciViewMatrix[0][0], ciViewMatrix[1][0], ciViewMatrix[2][0]);
		vec3 cameraUp = vec3(ciViewMatrix[0][1], ciViewMatrix[1][1], ciViewMatrix[2][1]);
		vec3 rightOffset = cameraRight * fScale * 0.5f * plantDefinitions[type].width * WIDTH_SCALE;
		vec3 upOffset = cameraUp * fScale * 0.5f * plantDefinitions[type].height;
		vec4 width = vec4(rightOffset, 0);
		vec4 height = vec4(upOffset, 0);

		mat4 rotation = makeYRotationMatrix(vRotation[0]);
		vec4 rotatedWidth = vec4(rightOffset, 0) * rotation;
		vec4 rotatedHeight = vec4(upOffset, 0) * rotation;

		vec3 cameraLocalOffset = (ciViewMatrixInverse[3].xyz - position.xyz);
		cameraLocalOffset = rotate(cameraLocalOffset, vec3(0, 1, 0), vRotation[0]);
		cameraEyePosition = (position.xyz + cameraLocalOffset);

		position = position + vec4(vec3(0, fScale * 0.5f * plantDefinitions[type].height, 0), 0);
		centrePosition = position.xyz;
		vec3 viewNormal = normalize(cameraEyePosition - position.xyz);

		const float octoBorder = 0.2f;
		const float borderRatio = (1.f / (1.f - (octoBorder * 2.f)));

		vec2 octoCoords = getHemiOctahedralCoordinatesFromNormal(viewNormal);
		vec2 _originalOctoCoords = octoCoords;
		_originalOctoCoords.x = clamp(((octoCoords.x  - octoBorder) * borderRatio), 0.f, 0.99f);
		_originalOctoCoords.y = clamp(((octoCoords.y  - octoBorder) * borderRatio), 0.f, 0.99f);
		octoCoords = vec2(_originalOctoCoords.x * (BIOME_BITMAP_X_SIZE - 1), _originalOctoCoords.y * (BIOME_BITMAP_Z_SIZE - 1));
		vec2 fractOcto = fract(octoCoords);
		vec2 baseOcto = floor(octoCoords);
		vec4 coords = texture(coordinatesTexture, vec2(fractOcto.x, 1.f - fractOcto.y));
		viewRatios = coords.xyz;

		int octoX = clamp(int(baseOcto.x), 0, BIOME_BITMAP_X_SIZE - 1);
		int octoZ = clamp(int(baseOcto.y), 0, BIOME_BITMAP_Z_SIZE - 1);
		views = ivec3(
			getOctoView(octoX, octoZ),  
			getOctoView(octoX + 1, octoZ + 1),
			coords.a > 0.5 ? getOctoView(octoX + 1, octoZ) : getOctoView(octoX, octoZ + 1)
		);

		if (INTERPOLATED_VIEWS < 3) {
			sortViews(views, viewRatios);
			viewRatios[2] = 0.f;
			if (INTERPOLATED_VIEWS < 2) {
				viewRatios[1] = 0.f;
			}
			viewRatios = additiveNormalise(viewRatios);
		}

		for (int i=0; i<4; i++) {
			vec2 vertexPosition = vertexPositions[i];
			vec4 worldPosition = position + (width * vertexPosition.x) + (height * vertexPosition.y);
			rotatedWorldPosition = (position + (rotatedWidth * vertexPosition.x) + (rotatedHeight * vertexPosition.y)).xyz;
			gl_Position = ciModelViewProjection * worldPosition;
			EmitVertex();
		}

		EndPrimitive();
	}
}
