#version 430

uniform mat4 ciModelViewProjection;
uniform ivec2 localOffset;
uniform ivec3 worldOffset;
uniform ivec3 cameraOffset;
uniform int resolution;
uniform int water_resolution;
uniform ivec2 water_size;
uniform ivec2 water_offset;
uniform sampler2DRect water0; // water row 0
uniform sampler2DRect waterOffset0; // water offset
uniform sampler2DRect ice0;
uniform sampler2DRect landHeight0;
uniform sampler2D perlinNoiseTexture;
uniform sampler2DRect lightingData; // lighting
uniform sampler2DRect shadowsData1; // shadows row 0
uniform sampler2DRect shadowsData2; // shadows row 1
uniform bool useWaterOffset;
uniform bool useTriangleSwitching;
uniform bool trianglesOpposite;
uniform bool shadowsEnabled;
uniform float DISPLAY_THRESHOLD;
uniform vec2 perlinNoiseSize;

in vec4 ciPosition;

out float discardVertex;
out float depth;
out float illumination;
out vec3 worldPosition;
out vec2 shadows1;
out vec2 shadows2;
out vec3 normal;

#include "common/constants.glsl"
#include "common/scale.glsl"

ivec2 getWaterPosition(vec2 position) {
	float _scale = scale(resolution, water_resolution);
	return ivec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

vec2 getDataPosition(vec2 position) {
	float _scale = scale(resolution, water_resolution);
	return vec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

void main() {
	vec4 vertexPosition = ciPosition;

	/* Read the water offset texture to figure out if we should flip the vertex ordering, so
		that the squares are divided into triangles the opposite way 
		We do this to avoid certain artifacts in the way the water is rendered, to make things smoother */
	if (useTriangleSwitching || trianglesOpposite) {
		int vertexNum = gl_VertexID % 6;
		if (vertexNum == 2 || vertexNum == 3) {
			if (vertexNum == 3) {
				vertexPosition -= vec4(1, 0, 0, 0);
			}
			if (vertexNum == 2) {
				vertexPosition -= vec4(0, 0, 1, 0);
			}
			vec2 waterPos = getWaterPosition(vertexPosition.xz) + vec2(0.5, 0.5);
			vec4 waterOffset = texture(waterOffset0, waterPos);
			bool offsetTextureTrianglesOpposite = (waterOffset.b == 1);
			if (trianglesOpposite) {
				if (useTriangleSwitching) {
					offsetTextureTrianglesOpposite = !offsetTextureTrianglesOpposite;
				} else {
					offsetTextureTrianglesOpposite = true;
				}
			}
			if (offsetTextureTrianglesOpposite) {
				if (vertexNum == 2) {
					vertexPosition += vec4(1, 0, 1, 0);
				}
			} else {
				if (vertexNum == 3) {
					vertexPosition += vec4(1, 0, 0, 0);
				}
				if (vertexNum == 2) {
					vertexPosition += vec4(0, 0, 1, 0);
				}
			}
		}	
	}

	/* Then figure out the actual vertex position */
	vec2 worldCoordinates = convertCoordinates(vertexPosition.xz, resolution, 1, vec2(0, 0), vec2(0, 0));
	vec4 _position = vec4(worldCoordinates.x, 0.0, worldCoordinates.y, 1.0);
	vec2 dataPos = getDataPosition(vertexPosition.xz) + vec2(0.5, 0.5);
	vec4 waterData = texture(water0, dataPos);
	vec4 iceData = texture(ice0, dataPos);
	vec4 landHeight = texture(landHeight0, dataPos);
	vec4 waterOffset = texture(waterOffset0, dataPos);
	illumination = texture(lightingData, dataPos)[0];

	if (shadowsEnabled) {
		vec2 _shadows = texture(shadowsData1, dataPos).xy;
		shadows1 = vec2(_shadows.x * 4096.f, _shadows.y * 32.f);
		_shadows = texture(shadowsData2, dataPos).xy;
		shadows2 = vec2(_shadows.x * 4096.f, _shadows.y * 32.f);
	}

	float iceHeight = iceData[0];
	if (iceHeight > 0.f && iceHeight < 1.f) {
		iceHeight = 1.f;
	}

	_position.y = landHeight[0] + iceHeight;

	if (useWaterOffset) {
		if (iceHeight > 0.f) {
			// a little bit of random variation to make things less uniform
			vec2 offset = vec2(-0.3f, -0.3f) + (texture(perlinNoiseTexture, dataPos / perlinNoiseSize).xy * 0.6f);
			_position += vec4(offset.x, 0.f, offset.y, 0.f) * water_resolution;
		} else {
			// but if the ice height is below the land, the offset needs to match otherwise ice will poke through
			_position += vec4(waterOffset.x, 0.0, waterOffset.y, 0.0) * water_resolution;
		}
	}

	if (dataPos.x < 0 || dataPos.y < 0 || dataPos.x > water_size.x || dataPos.y >= water_size.y) {
		discardVertex = 1;
	} else {
		discardVertex = 0;
	}

	worldPosition = worldOffset + _position.xyz;
	// If the ice display height is below the land, we need to make sure the height for calculating the shadow
	// isn't below the land, else we'll get a shadow border around the ice edge
	if (iceHeight < 0.f) {
		worldPosition.y -= iceHeight;
	}
	vec4 cameraPosition = vec4(cameraOffset, 0.0) + _position;
	gl_Position = ciModelViewProjection * cameraPosition;
	normal = vec3(landHeight[1], 1.f - (landHeight[1] + landHeight[2]), landHeight[2]);
}

