#version 430

uniform sampler2DRect biomes1;
uniform sampler2DRect biomes2;
uniform sampler2DRect grassData;
uniform sampler2D biomePalette;
uniform sampler2DArray surfaceTextures;
uniform sampler2DArray surfaceNormals;
uniform sampler2DArray biomeNoiseTexture;
uniform sampler3D rockDepths;
uniform float lightAdjust;
uniform bool renderCoverages;
uniform ivec2 water_size;
uniform float lightDirectionMix;
uniform vec3 diffuseColour;
uniform vec3 ambientColour;
uniform float diffuseAmount;
uniform float ambientAmount;
uniform float diffuseReflectionAmbient;
uniform float diffuseReflectionDiffuse;
uniform vec3 lightDirection;
uniform bool shadowsEnabled;
uniform int rockDepthsResolution;
uniform bool CALCULATE_DEPTHS;
uniform int textureSize;
uniform bool USE_TEXTURE_TILING_ENHANCEMENTS;
uniform bool diffuseLightingEnabled;
uniform float BUMP_MAPPING_AMBIENT_OCCLUSION;
uniform float BUMP_MAPPING_NORMAL_AMOUNT;
uniform float BUMP_MAPPING_DISPLACEMENT_HEIGHT;
uniform float BUMP_MAPPING_ITERATIONS;
uniform bool TRIPLANAR_TEXTURING;
uniform float TEXTURE_SIZE;

#include "common/interpolations.glsl"
#include "common/constants.glsl"
#include "common/data/biomes.glsl"
#include "common/noise.glsl"
#include "common/render/overlay.glsl"

in float discardFragment; // 1
in vec3 worldPosition;
in vec2 texCoords;
in vec2 shadows1; // 2
in vec2 shadows2;
in float transition; // 1
in float illumination; // ?1
in float groundwater;
in vec2 fDataPos;
in vec3 normal;

out vec4 _output;

#include "common/textures.glsl"

vec4 grassColour = vec4(139 / 255.f, 146 / 255.f, 80 / 255.f, 1.f);
vec4 marshColour = vec4(112 / 255.f, 69 / 255.f, 43 / 255.f, 1.f);
vec4 mudColour = vec4(85 / 255.f, 65 / 255.f, 46 / 255.f, 1.f);
vec4 sandColour = vec4(196 / 255.f, 157 / 255.f, 107 / 255.f, 1.f);
vec4 igneousColour = vec4(59 / 255.f, 58 / 255.f, 57 / 255.f, 1.f);
vec4 sandstoneColour = vec4(162 / 255.f, 102 / 255.f, 60 / 255.f, 1.f);
vec4 snowColour = vec4(205 / 255.f, 226 / 255.f, 242 / 255.f, 1.f);

float convertGrassCoverage(float value) {
	float diff = 1.f - value;
	diff = diff * diff * diff * diff;
	return 1.f - diff;
}

vec2 parallaxMapping(vec2 texCoords, int displacementTextureIndex, vec3 viewDirection) { 
    float layerDepth = 1.0 / BUMP_MAPPING_ITERATIONS;
    vec2 P = viewDirection.xy * BUMP_MAPPING_DISPLACEMENT_HEIGHT; 
    vec2 deltaTexCoords = P / BUMP_MAPPING_ITERATIONS;

  	vec2  currentTexCoords = texCoords;
	float currentDepthMapValue = texture(surfaceNormals, vec3(texCoords, displacementTextureIndex)).r;
	float currentLayerDepth = 0.f;

	while(currentLayerDepth < currentDepthMapValue)
	{
	    currentTexCoords -= deltaTexCoords;
	    currentDepthMapValue = texture(surfaceNormals, vec3(currentTexCoords, displacementTextureIndex)).r;  
	    currentLayerDepth += layerDepth;  
	}

	vec2 prevTexCoords = currentTexCoords + deltaTexCoords;
	float afterDepth  = currentDepthMapValue - currentLayerDepth;
	float beforeDepth = texture(surfaceNormals, vec3(prevTexCoords, displacementTextureIndex)).r - currentLayerDepth + layerDepth;
	float weight = afterDepth / (afterDepth - beforeDepth);
	vec2 finalTexCoords = prevTexCoords * weight + currentTexCoords * (1.0 - weight);

	return finalTexCoords;  
} 

vec4 textureSample(vec2 texCoords, int topIndex, int sideIndex) {
	if (TRIPLANAR_TEXTURING) {
		vec4 dx = texture(surfaceTextures, vec3(worldPosition.zy / TEXTURE_SIZE, sideIndex));
		vec4 dy = texture(surfaceTextures, vec3(worldPosition.xz / TEXTURE_SIZE, topIndex));
		vec4 dz = texture(surfaceTextures, vec3(worldPosition.xy / TEXTURE_SIZE, sideIndex));

		vec3 weights = abs(normal.xyz);
		weights = weights / (weights.x + weights.y + weights.z);

		return (dx * weights.x) + (dy * weights.y) + (dz * weights.z);
	} else {
		return texture(surfaceTextures, vec3(texCoords, topIndex));
	}
}

void getBiomeColour(int biomeId, vec4 biomeData, inout vec4 colour, inout float diffuse, float mipZoom, const BiomeLightingComponents biomeLightingComponents) {
	if (biomeId == 4 || biomeId == 15) {
		colour = vec4(0);
		diffuse = 0.7f;
		return;
	}
	
	vec4 biome1Lighting = getBiomeLighting2(biomeId, biomeLightingComponents);
	vec4 biomeTreeColour = texture(biomePalette, vec2(biomeData[2], (0.5 / 16.f) + (biomeId / 16.f)));
	vec4 biomeUnderlayColour = texture(biomePalette, vec2(biomeData[3], (0.5 / 16.f) + (biomeId / 16.f)));
	bool showTrunk = biomeTreeColour.a < 0.5f;
	biomeTreeColour.a = 1.f;
	if (biomeLightingComponents.snow) {
		biomeUnderlayColour = snowColour;
	}

	float treeCoverage = showTrunk ? biomeDefinitions[biomeId].winterTreeCoverage : biomeDefinitions[biomeId].normalTreeCoverage;

	vec4 noiseColour = vec4(0);
	float noiseDiffuse = 0.f;
	if (mipZoom < 1.f) {
		vec2 tetureWorldPos1 = (worldPosition.xz) / biomeDefinitions[biomeId].treeUnderlayNoiseDivisor;
		vec4 noiseValues1 = texture(noiseTexture, 
			vec2((tetureWorldPos1.x + 0.5f) / noiseTextureSize.x, (tetureWorldPos1.y + 0.5f) / noiseTextureSize.y)
		);

		vec2 tetureWorldPos2 = (worldPosition.xz) / biomeDefinitions[biomeId].treeLitNoiseDivisor;
		vec4 noiseValues2 = texture(noiseTexture, 
			vec2((tetureWorldPos2.x + 0.5f) / noiseTextureSize.x, (tetureWorldPos2.y + 0.5f) / noiseTextureSize.y)
		);

		if (noiseValues1[0] > treeCoverage) {
			noiseColour = biomeUnderlayColour;
			noiseDiffuse = biome1Lighting[2];
		} else {
			noiseColour = biomeTreeColour;
			if (noiseValues2[1] < biome1Lighting[0]) {
				noiseDiffuse = biome1Lighting[1];
			} else {
				noiseDiffuse = 0.f;
			}
		}
	}

	if (mipZoom > 0.f) {
		vec4 zoomedFinalColour =
			(biomeTreeColour * treeCoverage) + 
			(biomeUnderlayColour * (1.f - treeCoverage));
			
		float zoomedDiffuseLighting = (
			(biome1Lighting[2] * (1.f - treeCoverage)) +
			(biome1Lighting[0] * biome1Lighting[1] * treeCoverage)
		);

		colour = mix(noiseColour, zoomedFinalColour, mipZoom);
		diffuse = mix(noiseDiffuse, zoomedDiffuseLighting, mipZoom);
	} else {
		colour = noiseColour;
		diffuse = noiseDiffuse;
	}
}

void getBiomesUnderlay(vec2 dataPos, inout vec4 biome1, inout vec4 biome2, inout vec2 underlayCoverages) {
		vec2 _fractDataPos = fract(dataPos);
		underlayCoverages = vec2(0.f);

		vec4 _ids1 = textureGather(biomes1, floor(dataPos) + vec2(0.75f), 0);
		ivec4 ids1 = ivec4(_ids1 * 255);
		vec4 amounts1 = textureGather(biomes1, floor(dataPos) + vec2(0.75f), 1);

		vec4 _ids2 = textureGather(biomes2, floor(dataPos) + vec2(0.75f), 0);
		ivec4 ids2 = ivec4(_ids2 * 255);
		vec4 amounts2 = textureGather(biomes2, floor(dataPos) + vec2(0.75f), 1);

		// there are 16 biomes, but since we're doing all this to find out the biomes to render the biome underlay,
		// and we know that the last two biomes (riparian and nothing) don't have an underlay, we exclude them
		// this also fixes some issues with there being three biomes we're interpolating between in a square
		// near rivers which causes some abrupt edges in the underlay interpolations
		// other way around this would be to pull the highest three, rather than highest two biomes
		float counts[14] = float[14](0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		counts[ids1[0]] += (amounts1[0] * (1.f - _fractDataPos.x) * _fractDataPos.y);
		counts[ids1[1]] += (amounts1[1] * _fractDataPos.x * _fractDataPos.y);
		counts[ids1[2]] += (amounts1[2] * _fractDataPos.x * (1.f - _fractDataPos.y));
		counts[ids1[3]] += (amounts1[3] * (1.f - _fractDataPos.x) * (1.f - _fractDataPos.y));		

		counts[ids2[0]] += (amounts2[0] * (1.f - _fractDataPos.x) * _fractDataPos.y);
		counts[ids2[1]] += (amounts2[1] * _fractDataPos.x * _fractDataPos.y);
		counts[ids2[2]] += (amounts2[2] * _fractDataPos.x * (1.f - _fractDataPos.y));
		counts[ids2[3]] += (amounts2[3] * (1.f - _fractDataPos.x) * (1.f - _fractDataPos.y));

		biome1 = vec4(0);
		biome2 = vec4(0);

		for (int i=0; i<14; i++) {
			underlayCoverages[biomeDefinitions[i].underlayIndex] += counts[i];
			if (counts[i] > biome1[1]) {
				biome2 = biome1;
				biome1 = vec4(i / 255.f, counts[i], 0.f, 0.6f);
			} else {
				if (counts[i] > biome2[1]) {
					biome2 = vec4(i / 255.f, counts[i], 0.f, 0.6f);
				}
			}
		}

		biome1[1] = clamp(biome1[1], 0.f, 1.f);
		biome2[1] = min(biome2[1], 1.f - biome1[1]);
}

void main() {
	if (discardFragment > 0) {
		discard;
	}

	vec4 finalColour;

	// Only needed for dispacement mapping
	// vec3 tangent = vec3(normal.y, -normal.x, normal.z);
	// vec3 bitangent = vec3(normal.x, -normal.z, normal.y);
	// mat3 TBN = mat3(normalize(tangent), normalize(bitangent), normalize(normal));

	const float rockDepthsSize = 4096.f / rockDepthsResolution;
	float yPos = ((worldPosition.y / rockDepthsResolution) + 0.5f) / rockDepthsSize;
	vec3 rockDepthsPosition = vec3((fDataPos.x + 0.5f) / float(water_size.x), yPos, (fDataPos.y) / float(water_size.y));
	vec4 values = texture(rockDepths, rockDepthsPosition);

	float dirtCoverage = clamp(values[0], 0.f, 1.f);
	float igneousCoverage = clamp(values[1], 0.f, 1.f);
	float sandstoneCoverage = clamp(values[2], 0.f, 1.f);
	float sandCoverage = clamp(2.f - worldPosition.y, 0.f, 1.f);

	dirtCoverage += (1.f - (dirtCoverage + igneousCoverage + sandstoneCoverage));

	vec4 grassColour;
	vec4 grassData = texture(grassData, fDataPos + vec2(0.5f));
	float grassCoverage = grassData[0];
	float snowCoverage = ((grassData[2] > 0.5f) ? 1.f : 0.f);
	if (grassCoverage > 0) {
		grassColour = texture(biomePalette, vec2(grassData[1], (4.5f / 16.f)));
		grassColour = mix(grassColour, marshColour, grassData[3] * 0.3f * (1.f - snowCoverage));
		grassCoverage = exponentialOut(grassCoverage);
	}

	float _shadows = 1;
	if (shadowsEnabled) {
		float _shadows1 = 1.f;
		if (worldPosition.y < shadows1[0]) {
			float shadowDist = shadows1[1];
			float shadowBottom = shadows1[0] - shadowDist;
			if (worldPosition.y < shadowBottom) {
				_shadows1 = 0.f;
			} else {
				_shadows1 = mix(1.0, 0.f, (shadows1[0] - worldPosition.y) / shadowDist);
			}
		}

		float _shadows2 = 1.f;
		if (worldPosition.y < shadows2[0]) {
			float shadowDist = shadows2[1];
			float shadowBottom = shadows2[0] - shadowDist;
			if (worldPosition.y < shadowBottom) {
				_shadows2 = 0.f;
			} else {
				_shadows2 = mix(1.0, 0.f, (shadows2[0] - worldPosition.y) / shadowDist);
			}
		}
		_shadows = mix(_shadows1, _shadows2, lightDirectionMix);
	}

	vec2 _fDataPos = fDataPos;
	vec2 fractDataPos = fract(_fDataPos);
	int biomeDataPosX, biomeDataPosY;
	float diffuseMultiplier = 1.f;
	float biomeNoiseMult = 0.6f;

	// Render zoomed out
	if (transition > 0.f) {
		vec4 biomeColour = vec4(0);
		float biomeDiffuse = 0.f;
		float _lod = textureQueryLod(biomeNoiseTexture, worldPosition.xz / noiseTextureSize).x;
		// Transition between dithered rendering and smoothed rendering
		float smoothedRenderAmount = clamp((_lod + 0.25f) / 2.f, 0.f, 1.f);
		// Transition between interpolating between 4 closest data points and just picking one
		float singleDataRenderAmount = clamp(_lod - 1.75f, 0.f, 1.f);
		vec4 biomeColourSmooth = vec4(0), biomeColourDithered = vec4(0);
		float biomeDiffuseSmooth = 0.f, biomeDiffuseDithered = 0.f;

		vec4 rockColour = vec4(0);
		rockColour += igneousColour * igneousCoverage;
		rockColour += sandstoneColour * sandstoneCoverage;
		vec4 dirtColour = mix(sandColour, mudColour, groundwater);
		dirtColour = mix(dirtColour, sandColour, sandCoverage);
		vec4 groundColour = rockColour + (dirtColour * dirtCoverage);
		groundColour = mix(groundColour, snowColour, snowCoverage);


		const BiomeLightingComponents biomeLightingComponents = getLightingComponents(worldPosition.xyz, (snowCoverage == 1.f));
		float ditheredMix = 1.f;

		// Undithered view when we're zoomed far enough that dithering would cause swimming when moving
		if (smoothedRenderAmount > 0.f) {
			vec4 _biome, tempBiomeColour;
			float tempBiomeDiffuse;
			float totalContribution = 0.f;
			ivec2 biomeDataPos = ivec2(floor(_fDataPos));

			_biome = texelFetch(biomes1, biomeDataPos);
			int id = int(_biome[0] * 255.f);
			getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);

			// When really far zoomed out, don't need to interpolate between the four nearest data points- just choose one
			if (singleDataRenderAmount > 0.f) {
				float contribution = _biome[1] * singleDataRenderAmount;
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;	
			}

			// Otherwise interpolate between the four data surrounding data points
			if (singleDataRenderAmount < 1.f) {
				float contribution = _biome[1] * (1.f - fractDataPos.x) * (1.f - fractDataPos.y) * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;	

				_biome = texelFetch(biomes1, biomeDataPos + ivec2(1, 0));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * fractDataPos.x * (1.f - fractDataPos.y) * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;

				_biome = texelFetch(biomes1, biomeDataPos + ivec2(1, 1));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * fractDataPos.x * fractDataPos.y * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;

				_biome = texelFetch(biomes1, biomeDataPos + ivec2(0, 1));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * (1.f - fractDataPos.x) * fractDataPos.y * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;
			}

			_biome = texelFetch(biomes2, biomeDataPos);
			id = int(_biome[0] * 255.f);
			getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);

			// When really far zoomed out, don't need to interpolate between the four nearest data points- just choose one
			if (singleDataRenderAmount > 0.f) {
				float contribution = _biome[1] * singleDataRenderAmount;
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;	
			}

			// Otherwise interpolate between the four data surrounding data points
			if (singleDataRenderAmount < 1.f) {
				float contribution = _biome[1] * (1.f - fractDataPos.x) * (1.f - fractDataPos.y) * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;	

				_biome = texelFetch(biomes2, biomeDataPos + ivec2(1, 0));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * fractDataPos.x * (1.f - fractDataPos.y) * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;

				_biome = texelFetch(biomes2, biomeDataPos + ivec2(1, 1));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * fractDataPos.x * fractDataPos.y * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;

				_biome = texelFetch(biomes2, biomeDataPos + ivec2(0, 1));
				id = int(_biome[0] * 255.f);
				getBiomeColour(id, _biome, tempBiomeColour, tempBiomeDiffuse, 1.f, biomeLightingComponents);
				contribution = _biome[1] * (1.f - fractDataPos.x) * fractDataPos.y * (1.f - singleDataRenderAmount);	
				totalContribution += contribution;
				biomeColourSmooth += tempBiomeColour * contribution;
				biomeDiffuseSmooth += tempBiomeDiffuse * contribution;
			}

			biomeDiffuseSmooth += (1.f - totalContribution);

			if (biomeColourSmooth.a > 0) {
				biomeColourSmooth.xyz = biomeColourSmooth.xyz / biomeColourSmooth.a;
			}
			ditheredMix = 1.f - smoothedRenderAmount;
		} 

		if (ditheredMix > 0.f) {

			vec4 biomeNoise = texture(noiseTexture, 
				vec2(((worldPosition.x * biomeNoiseMult) + 16.f) / noiseTextureSize.x, ((worldPosition.z * biomeNoiseMult) + 16.f) / noiseTextureSize.y)
			);
			vec4 biome;
			int biomeId;

			if (biomeNoise[1] > fractDataPos.x) {
				biomeDataPosX = int(floor(_fDataPos.x));
			} else {
				biomeDataPosX = int(ceil(_fDataPos.x));
			}
			if (biomeNoise[2] > fractDataPos.y) {
				biomeDataPosY = int(floor(_fDataPos.y));
			} else {
				biomeDataPosY = int(ceil(_fDataPos.y));
			}

			ivec2 biomeDataPos = ivec2(biomeDataPosX, biomeDataPosY);

			vec4 biome1 = texelFetch(biomes1, biomeDataPos);
			vec4 biome2 = texelFetch(biomes2, biomeDataPos);

			if (biomeNoise[0] <= biome1[1]) {
				biomeId = int(biome1[0] * 255.f);
				biome = biome1;
			} else {
				if (biomeNoise[0] > (1.f - biome2[1])) {
					biomeId = int(biome2[0] * 255.f);
					biome = biome2;
				} else {
					biomeId = 15;
					biome = vec4(15 / 255.f, 0.f, 0.f, 0.f);
				}
			}

			getBiomeColour(biomeId, biome, biomeColourDithered, biomeDiffuseDithered, smoothedRenderAmount, biomeLightingComponents);
		}

		finalColour = mix(groundColour, grassColour, grassCoverage);
		biomeColour = mix(biomeColourSmooth, biomeColourDithered, ditheredMix);
		finalColour = vec4(mix(finalColour, biomeColour, biomeColour.a).xyz, 1.f);

		biomeDiffuse = mix(biomeDiffuseSmooth, biomeDiffuseDithered, ditheredMix);
		diffuseMultiplier = mix(diffuseMultiplier, biomeDiffuse, biomeColour.a * transition);
	}
	
	vec4 bumpNormal = vec4(0.5, 0.5, 1, 1);

	// Render zoomed in
	if (transition < 0.99f) {

		vec4 biome1, biome2;
		vec2 biomeUnderlayCoverages;
		getBiomesUnderlay(fDataPos, biome1, biome2, biomeUnderlayCoverages);

// Displacement mapping
#if 0
		int colourIndex = 0;
		int normalsIndex = 0;
		int displacementIndex = 1;

		vec4 rockColour = vec4(0);
		vec4 rockNormal = vec4(0);

		vec2 displacedTexCoords = parallaxMapping(texCoords, displacementIndex, normalize(TBN * (eyePoint - _worldPosition)));

		vec4 groundColour = texture(surfaceTextures, vec3(displacedTexCoords, colourIndex));
		vec4 groundNormal = texture(surfaceNormals, vec3(displacedTexCoords, normalsIndex));

		bumpNormal = groundNormal;
		groundColour = vec4(groundColour.rgb * mix(1.f, groundColour.a, BUMP_MAPPING_AMBIENT_OCCLUSION), 1.f);
#else
		const int igneousIndex = 3, sandstoneIndexTop = 2, sandstoneIndexSide = 4, sandIndex = 1, mudIndex = 0;
		vec4 rockColour = vec4(0);
		vec4 rockNormal = vec4(0);

		rockColour += textureSample(texCoords, igneousIndex, igneousIndex) * igneousCoverage;
		rockColour += textureSample(texCoords, sandstoneIndexTop, sandstoneIndexTop) * sandstoneCoverage;
		vec4 dirtColour = mix(textureSample(texCoords, sandIndex, sandIndex), textureSample(texCoords, mudIndex, mudIndex), groundwater);
		dirtColour = mix(dirtColour, textureSample(texCoords, sandIndex, sandIndex), sandCoverage);
		vec4 groundColour = rockColour + (dirtColour * dirtCoverage);
#endif

		// Right now we only have one underlay that actually renders
		// The vec2 biomeUnderCoverages is the coverage of each underlay, 
		// biomeCoverages[0] is no underlay
		// biomeCoverages[1] is the forest underlay
		// If we add more underlays, it'll need to be changed to a vec3 or a vec4
		// The Biomes SSBO has an underlayIndex which is the index into biomeCoverages
		// Then we manually map that to an index of surfaceTextures
		const int underlayIndex = 5;
		vec4 biomeUnderlay = texture(surfaceTextures, vec3(texCoords, underlayIndex));
		vec4 _finalColour = mix(groundColour, grassColour, grassCoverage);
		_finalColour = mix(_finalColour, biomeUnderlay, biomeUnderlayCoverages[1]);
		_finalColour = mix(_finalColour, snowColour, snowCoverage);
		finalColour = mix(finalColour, _finalColour, 1.f - transition);
	}

	vec3 _normal = normal;

#if 0
	if (BUMP_MAPPING_NORMAL_AMOUNT > 0.f) {
		bumpNormal = bumpNormal * 2.0 - 1.0;
		vec3 transformedBumpNormal = normalize(TBN * bumpNormal.xyz);
		_normal = normalize(normal + (BUMP_MAPPING_NORMAL_AMOUNT * transformedBumpNormal));
	}
#endif

	float diffuseReflection = 1.f;
	if (diffuseLightingEnabled) {
		diffuseReflection = clamp(dot(_normal, vec3(lightDirection.x, -lightDirection.y, lightDirection.z)), 0.f, 1.f);
	}

	float diffuseLighting = clamp((illumination * _shadows * diffuseAmount * mix(1.f, diffuseReflection, diffuseReflectionDiffuse)), 0.f, 1.f);
	float ambientLighting = clamp(illumination * ambientAmount * mix(1.f, diffuseReflection, diffuseReflectionAmbient), 0.f, 1.f);
	diffuseLighting = diffuseLighting * diffuseMultiplier;

	_output = vec4(
		lightAdjust * (
			(finalColour.xyz * diffuseLighting * diffuseColour) + 
			(finalColour.xyz * ambientLighting * ambientColour)
		), 
		1.f
	);	

	_output = renderOverlay(_output, worldPosition, transition > 0.99f);
}
