#version 150

uniform mat4 ciModelViewProjection;
uniform float circumference;
uniform ivec2 localOffset;
uniform ivec3 worldOffset;
uniform ivec3 cameraOffset;
uniform int resolution;
uniform float fResolution;
uniform int data1_resolution;
uniform ivec2 dataOffset;
uniform sampler2D mainTexture;
uniform sampler2DRect data1; // water row 0
uniform sampler2DRect data2; // water offset
uniform sampler2DRect lightingData; // land shading
uniform sampler2DRect data4; // lava row 0
uniform sampler2DRect data5; // lava row 2
uniform sampler2DRect shadowsData1; // shadows row 0
uniform sampler2DRect shadowsData2; // shadows row 1
uniform sampler2DRect landHeight1; // landHeight row 0
uniform sampler2D noiseTexture;
uniform ivec2 data1_size;
uniform bool useWaterOffset;
uniform bool useTriangleSwitching;
uniform bool trianglesOpposite;
uniform bool shadowsEnabled;

in vec4 ciPosition;

out float discardVertex;
out float illumination;
out float temperatureBand;
out vec2 shadows1;
out vec2 shadows2;
out vec3 worldPosition;
out vec2 texCoords[4];
out vec3 normal;

float scale(int resolution1, int resolution2) {
	float ret = 1.f;
	if (resolution1 < 0) {
		ret = ret / -resolution1;
	} else {
		ret = ret * resolution1;
	}
	if (resolution2 < 0) {
		ret = ret * -resolution2;
	} else {
		ret = ret / resolution2;
	}
	return ret;
}

vec2 convertCoordinates(vec2 coordinates, int inputResolution, int outputResolution, vec2 inputOffset, vec2 outputOffset) {
	float _scale = scale(inputResolution, outputResolution);
	return vec2((coordinates.x * _scale) + (inputOffset.x * _scale) - (outputOffset.x),
				(coordinates.y * _scale) + (inputOffset.y * _scale) - (outputOffset.y));
}

vec4 getPosition(vec4 originalPosition) {
	return vec4((originalPosition.x * fResolution) + localOffset.x, 0.0, (originalPosition.z * fResolution) + localOffset.y, 1.0);
}

ivec2 getWaterPosition(vec2 position) {
	float _scale = scale(resolution, data1_resolution);
	return ivec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

vec2 getDataPosition(vec2 position) {
	float _scale = scale(resolution, data1_resolution);
	return vec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

void main() {
	vec4 vertexPosition = ciPosition;

	/* Read the water offset texture to figure out if we should flip the vertex ordering, so
		that the squares are divided into triangles the opposite way 
		We do this to avoid certain artifacts in the way the water is rendered, to make things smoother */
	if (useTriangleSwitching || trianglesOpposite) {
		int vertexNum = gl_VertexID % 6;
		if (vertexNum == 2 || vertexNum == 3) {
			if (vertexNum == 3) {
				vertexPosition -= vec4(1, 0, 0, 0);
			}
			if (vertexNum == 2) {
				vertexPosition -= vec4(0, 0, 1, 0);
			}
			vec2 waterPos = getWaterPosition(vertexPosition.xz) + vec2(0.5, 0.5);
			vec4 waterOffset = texture(data2, waterPos);
			bool offsetTextureTrianglesOpposite = (waterOffset.b == 1);
			if (trianglesOpposite) {
				if (useTriangleSwitching) {
					offsetTextureTrianglesOpposite = !offsetTextureTrianglesOpposite;
				} else {
					offsetTextureTrianglesOpposite = true;
				}
			}
			if (offsetTextureTrianglesOpposite) {
				if (vertexNum == 2) {
					vertexPosition += vec4(1, 0, 1, 0);
				}
			} else {
				if (vertexNum == 3) {
					vertexPosition += vec4(1, 0, 0, 0);
				}
				if (vertexNum == 2) {
					vertexPosition += vec4(0, 0, 1, 0);
				}
			}
		}	
	}

	vec2 dataPos = getDataPosition(vertexPosition.xz) + vec2(0.5, 0.5);
	vec4 lavaDisplay = texture(data5, dataPos);

	discardVertex = 0;
	if (dataPos.x < 0 || dataPos.y < 0 || dataPos.x >= data1_size.x || dataPos.y >= data1_size.y || lavaDisplay[2] == -3.f) {
		gl_Position = vec4(-2.f);
		discardVertex = 1;
		return;
	}

	vec4 water = texture(data1, dataPos);
	vec4 waterOffset = texture(data2, dataPos);
	illumination = texture(lightingData, dataPos)[0];
	vec4 lava0 = texture(data4, dataPos);

	if (shadowsEnabled) {
		vec2 shadows = texelFetch(shadowsData1, ivec2(dataPos)).xy;
		shadows1 = vec2(shadows.x * 4096.f, shadows.y * 32.f);
		shadows = texelFetch(shadowsData2, ivec2(dataPos)).xy;
		shadows2 = vec2(shadows.x * 4096.f, shadows.y * 32.f);
	}

	vec2 worldCoordinates = convertCoordinates(vertexPosition.xz, resolution, 1, vec2(0), vec2(0));
	vec4 _position = vec4(worldCoordinates.x, water[0] + lavaDisplay[2], worldCoordinates.y, 1.0);
	if (useWaterOffset) {
		_position += vec4(waterOffset.x, 0.0, waterOffset.y, 0.0) * data1_resolution;
	}
	worldPosition = worldOffset + _position.xyz;
	vec4 cameraPosition = vec4(cameraOffset, 0.f) + _position;
	gl_Position = ciModelViewProjection * cameraPosition;

	texCoords[0] = vec2((worldPosition.x + (lavaDisplay[0]) * 16.f) / 32.f, (worldPosition.z + (lavaDisplay[1]) * 16.f) / 32.f);
	texCoords[1] = vec2((worldPosition.x + (fract(lavaDisplay[0]) * 16.f)) / 16.f, (worldPosition.z + (fract(lavaDisplay[1]) * 16.f)) / 16.f);
	texCoords[2] = texCoords[1];
	texCoords[3] = vec2(worldPosition.x / 3.f, worldPosition.z / 3.f);

	temperatureBand = clamp(3.f - (((lavaDisplay[3] - 300.f) / 700.f) * 3.f), 0.f, 2.99f);

	vec4 landHeight = texture(landHeight1, dataPos);
	normal = vec3(landHeight[1], 1.f - (landHeight[1] + landHeight[2]), landHeight[2]);
}

