#version 430

uniform float DEPTH_CUTOFF;
uniform vec3 diffuseColour;
uniform vec3 ambientColour;
uniform float diffuseAmount;
uniform float ambientAmount;
uniform float lightDirectionMix;
uniform bool shadowsEnabled;
uniform sampler2D previousDepth;
uniform sampler2D waterGradient;
uniform sampler2D previousColour;
uniform sampler2D perlinNoiseTexture;
uniform sampler2D waterFlowTexture;
uniform sampler2DRect biomes1;
uniform sampler2DRect biomes2;
uniform ivec2 screenSize;
uniform ivec3 cameraGlobalOffset;
uniform float nearPlane;
uniform float farPlane;
uniform mat4 ciViewMatrixInverse;
uniform mat4 ciProjectionMatrixInverse;

uniform int stepCount;
uniform float REFRACTION_SIZE;
uniform float REFRACTION_SPEED;
uniform float REFRACTION_AMOUNT;
uniform bool USE_REFRACTION;

#include "common/interpolations.glsl"
#include "common/water_display.glsl"
#include "common/render/overlay.glsl"

float linearizeDepth(float zBufferValue) {
    // With glClipControl(GL_LOWER_LEFT, GL_ZERO_TO_ONE) and reversed z-depth:
    // - zBufferValue is already in [0, 1] range
    // - Near objects have high z values (close to 1), far objects have low z values (close to 0)
    // - For reversed z, we need to invert before applying the standard linearization formula
    float reversedDepth = 1.0 - zBufferValue;
    float ndcDepth = reversedDepth * 2.0 - 1.0; // Convert [0, 1] to [-1, 1]
    return (2.0 * nearPlane * farPlane) / (farPlane + nearPlane - ndcDepth * (farPlane - nearPlane));
}

in float discardVertex;
in float depth;
in float illumination;
in vec3 worldPosition;
in vec2 shadows1;
in vec2 shadows2;
in vec2 flowDirection;
in float flowSpeed;
in float overrideAlpha;

out vec4 _output;

vec3 depthToWorldPosition(float depth) {
	vec4 ndcPosition = vec4(
	    (gl_FragCoord.x / screenSize.x) * 2.0 - 1.0,
	    (gl_FragCoord.y / screenSize.y) * 2.0 - 1.0,
	    // With glClipControl(GL_LOWER_LEFT, GL_ZERO_TO_ONE), depth is already in [0,1]
	    // With reversed z-depth, we need to invert then map to [-1,1]
	    (1.0 - depth) * 2.0 - 1.0,
	    1.0
	);
	vec4 viewSpacePosition = ciProjectionMatrixInverse * ndcPosition;
	viewSpacePosition /= viewSpacePosition.w; // Perspective divide

	return vec3(cameraGlobalOffset) + (ciViewMatrixInverse * viewSpacePosition).rgb;
}

void main() {
	if (discardVertex > 0 || depth < -1.99f) {
		discard;
	}

	float shadows = 1.f;

	if (shadowsEnabled) {
		float _shadows1 = 1.f;
		if (worldPosition.y < shadows1[0]) {
			float shadowDist = shadows1[1];
			float shadowBottom = shadows1[0] - shadowDist;
			if (worldPosition.y < shadowBottom) {
				_shadows1 = 0.f;
			} else {
				_shadows1 = mix(1.0, 0.f, (shadows1[0] - worldPosition.y) / shadowDist);
			}
		}

		float _shadows2 = 1.f;
		if (worldPosition.y < shadows2[0]) {
			float shadowDist = shadows2[1];
			float shadowBottom = shadows2[0] - shadowDist;
			if (worldPosition.y < shadowBottom) {
				_shadows2 = 0.f;
			} else {
				_shadows2 = mix(1.0, 0.f, (shadows2[0] - worldPosition.y) / shadowDist);
			}
		}

		shadows = mix(_shadows1, _shadows2, lightDirectionMix);
	}

	float linearDepth = linearizeDepth(gl_FragCoord.z);
	vec2 existingBufferTexcoords = gl_FragCoord.xy;
	vec2 refraction = vec2(0,0);
	if (USE_REFRACTION && overrideAlpha < 0.8f) {
		refraction = (texture(perlinNoiseTexture, vec2(
			(worldPosition.x / REFRACTION_SIZE) + (stepCount / REFRACTION_SPEED),
			(worldPosition.z / REFRACTION_SIZE) + (stepCount / REFRACTION_SPEED)
		)).xy - vec2(0.5, 0.5)) * REFRACTION_AMOUNT / max(1.f, linearDepth);	
	}	
	existingBufferTexcoords += refraction;
	existingBufferTexcoords = clamp(existingBufferTexcoords, vec2(0, 0), vec2(screenSize) - vec2(1, 1));

	vec4 existingPixel = texelFetch(previousColour, ivec2(existingBufferTexcoords), 0);
	// definitely something wrong with this- I suspect both linearizeDepth and depthToWorldPosition are wrong since changing to reversed z depth
	// float previousDepthValue = texelFetch(previousDepth, ivec2(gl_FragCoord.xy), 0)[0];
	// vec3 previousWorldPosition = depthToWorldPosition(previousDepthValue);
	// float correctDepth = max(depth, worldPosition.y - previousWorldPosition.y);
	float correctDepth = depth;

	float colourDepth = getWaterColourTextureCoordinate(correctDepth);
	float alphaDepth = getWaterAlpha(correctDepth, overrideAlpha);
	vec3 colour = texture(waterGradient, vec2(colourDepth, 0.5f)).rgb;

	float diffuseLighting = clamp((illumination * shadows * diffuseAmount), 0.f, 1.f);
	float ambientLighting = clamp(illumination * ambientAmount, 0.f, 1.f);

	vec2 _flowDirection = flowDirection;
	float fs = 32.f;
	_flowDirection.y = round(_flowDirection.y * 2) / 2.f;
	_flowDirection.x = (_flowDirection.x < 0 ? -1.f : 1.f) * (1.f - abs(_flowDirection.y));

#if 0
	vec2 flowCoordsX = fract(vec2(
		worldPosition.x / fs,
		(worldPosition.x / fs) + ((stepCount / 128.f) * -sign(_flowDirection.x))
	));
	vec2 flowCoordsZ = fract(vec2(
		worldPosition.z / fs,
		(worldPosition.z / fs) + ((stepCount / 128.f) * -sign(_flowDirection.y))
	));
	vec4 flowTextureX = texture(waterFlowTexture, flowCoordsX);
	vec4 flowTextureZ = texture(waterFlowTexture, flowCoordsZ);
	vec4 flowTexture = mix(flowTextureX, flowTextureZ, abs(_flowDirection.y));
#else
	vec2 flowCoordsX = fract(vec2(
		worldPosition.z / fs,
		(worldPosition.x / fs) + (stepCount / 128.f) * -sign(_flowDirection.x)
	));
	vec2 flowCoordsZ = fract(vec2(
		worldPosition.x / fs,
		(-worldPosition.z / fs) + (stepCount / 128.f) * -sign(_flowDirection.y)
	));
	vec2 flowCoords = vec2(
		mix(flowCoordsX.x, flowCoordsZ.x, abs(_flowDirection.y)), 
		mix(flowCoordsX.y, flowCoordsZ.y, abs(_flowDirection.y))
	);
	vec4 flowTexture = texture(waterFlowTexture, flowCoords);
#endif 

	vec4 water = vec4(
		(
			(colour * diffuseLighting * diffuseColour) + 
			(colour * ambientLighting * ambientColour)
		), 
		clamp(alphaDepth, 0.f, 1.f)
	);

	// water = mix(water, flowTexture, flowTexture.a * 0.5f);
	_output = vec4(mix(existingPixel, water, water.a).rgb, 1.f);

	// _output = vec4(flowCoords.x, flowCoords.y, 0.f, 1.f);
	// _output = vec4(abs(_flowDirection.x), abs(_flowDirection.y), 0.f, 1.f);

	_output = renderOverlay(_output, worldPosition, true);

	// _output = vec4(fract(previousWorldPosition.y / 1024.f), fract(previousWorldPosition.y / 128.f), fract(previousWorldPosition.y / 16.f), 1.f);
}
