#version 430

uniform mat4 ciModelViewProjection;
uniform ivec2 localOffset;
uniform ivec3 worldOffset;
uniform ivec3 cameraOffset;
uniform int resolution;
uniform int data1_resolution;
uniform ivec2 dataOffset;
uniform sampler2DRect data1; // water row 0
uniform sampler2DRect data2; // water offset
uniform sampler2DRect water2; // water offset
uniform sampler2DRect landHeight0;
uniform sampler2DRect lightingData; // lighting
uniform sampler2DRect shadowsData1; // shadows row 0
uniform sampler2DRect shadowsData2; // shadows row 1
uniform ivec2 data1_size;
uniform bool useWaterOffset;
uniform bool useTriangleSwitching;
uniform bool trianglesOpposite;
uniform bool shadowsEnabled;
uniform vec3 eyePoint;

in vec4 ciPosition;

out float discardVertex;
out float depth;
out float illumination;
out vec3 worldPosition;
out vec2 shadows1;
out vec2 shadows2;
out vec2 flowDirection;
out float flowSpeed;
out float overrideAlpha;

#define M_PI 3.14159265358979323846

highp float rand(vec2 co) {

	int x = int(co.x);
	int y = int(co.y);
	highp float result = 0.f;

	for (int i = 0; i < 4; i++) {
		x += (x << 29);
		x ^= (x << 11);
		x ^= (y << 17);
		x ^= (x >> 19);
		x ^= (y >> 6);
		x ^= (x << 4);
	}

	result = x;
	
	return (x & 8191) / 8192.0;
}

float scale(int resolution1, int resolution2) {
	float ret = 1.f;
	if (resolution1 < 0) {
		ret = ret / -resolution1;
	} else {
		ret = ret * resolution1;
	}
	if (resolution2 < 0) {
		ret = ret * -resolution2;
	} else {
		ret = ret / resolution2;
	}
	return ret;
}

vec2 convertCoordinates(vec2 coordinates, int inputResolution, int outputResolution, vec2 inputOffset, vec2 outputOffset) {
	float _scale = scale(inputResolution, outputResolution);
	return vec2((coordinates.x * _scale) + (inputOffset.x * _scale) - (outputOffset.x),
				(coordinates.y * _scale) + (inputOffset.y * _scale) - (outputOffset.y));
}

ivec2 getWaterPosition(vec2 position) {
	float _scale = scale(resolution, data1_resolution);
	return ivec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

vec2 getDataPosition(vec2 position) {
	float _scale = scale(resolution, data1_resolution);
	return vec2((position.x * _scale) + (localOffset.x * _scale), (position.y * _scale) + (localOffset.y * _scale));
}

void main() {
	vec4 vertexPosition = ciPosition;

	/* Read the water offset texture to figure out if we should flip the vertex ordering, so
		that the squares are divided into triangles the opposite way 
		We do this to avoid certain artifacts in the way the water is rendered, to make things smoother */
	if (useTriangleSwitching || trianglesOpposite) {
		int vertexNum = gl_VertexID % 6;
		if (vertexNum == 2 || vertexNum == 3) {
			if (vertexNum == 3) {
				vertexPosition -= vec4(1, 0, 0, 0);
			}
			if (vertexNum == 2) {
				vertexPosition -= vec4(0, 0, 1, 0);
			}
			vec2 waterPos = getWaterPosition(vertexPosition.xz) + vec2(0.5, 0.5);
			vec4 waterOffset = texture(data2, waterPos);
			bool offsetTextureTrianglesOpposite = (waterOffset.b == 1);
			if (trianglesOpposite) {
				if (useTriangleSwitching) {
					offsetTextureTrianglesOpposite = !offsetTextureTrianglesOpposite;
				} else {
					offsetTextureTrianglesOpposite = true;
				}
			}
			if (offsetTextureTrianglesOpposite) {
				if (vertexNum == 2) {
					vertexPosition += vec4(1, 0, 1, 0);
				}
			} else {
				if (vertexNum == 3) {
					vertexPosition += vec4(1, 0, 0, 0);
				}
				if (vertexNum == 2) {
					vertexPosition += vec4(0, 0, 1, 0);
				}
			}
		}	
	}

	/* Then figure out the actual vertex position */
	vec2 worldCoordinates = convertCoordinates(vertexPosition.xz, resolution, 1, vec2(0, 0), vec2(0, 0));
	vec4 _position = vec4(worldCoordinates.x, 0.0, worldCoordinates.y, 1.0);
	vec2 dataPos = getDataPosition(vertexPosition.xz) + vec2(0.5, 0.5);
	vec4 data = texture(data1, dataPos);
	vec4 water2Data = texture(water2, dataPos);
	vec4 waterOffset = texture(data2, dataPos);
	vec4 landHeight = texture(landHeight0, dataPos);
	illumination = texture(lightingData, dataPos)[0];

	if (shadowsEnabled) {
		vec2 _shadows = texelFetch(shadowsData1, ivec2(dataPos)).xy;
		shadows1 = vec2(_shadows.x * 4096.f, _shadows.y * 32.f);
		_shadows = texelFetch(shadowsData2, ivec2(dataPos)).xy;
		shadows2 = vec2(_shadows.x * 4096.f, _shadows.y * 32.f);
	}

	_position.y = max(0.0, landHeight.r + data.a);

	if (useWaterOffset) {
		_position += vec4(waterOffset.x, 0.0, waterOffset.y, 0.0) * data1_resolution;
	}

	if (dataPos.x < 0 || dataPos.y < 0 || dataPos.x >= data1_size.x || dataPos.y >= data1_size.y) {
		discardVertex = 1;
	} else {
		discardVertex = 0;
	}
	flowDirection = vec2(0, 0);
	int _flowDirection = int(water2Data[0]);
	if (_flowDirection == 0) {
		flowDirection = vec2(0, -1);
	}
	if (_flowDirection == 1) {
		flowDirection = vec2(1, 0);
	}
	if (_flowDirection == 2) {
		flowDirection = vec2(0, 1);
	}
	if (_flowDirection == 3) {
		flowDirection = vec2(-1, 0);
	}

	worldPosition = worldOffset + _position.xyz;
	vec4 cameraPosition = vec4(cameraOffset, 0.0) + _position;
	gl_Position = ciModelViewProjection * cameraPosition;
	depth = -min(0, landHeight.r) + data.a;
	flowSpeed = 1.f;
	float eyeDistance = distance(eyePoint, cameraPosition.xyz);
	overrideAlpha = clamp((eyeDistance - 256.f) / 256.f, 0.f, 0.8f);
}

