// Fragment shader for the GPU water flow simulation pass that calculates occlusion and diffuse lighting for land
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;

layout (binding = 0, r8) uniform restrict image2DRect lighting0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect debug0Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect debug1Write;

uniform ivec2 water_size;
uniform ivec2 OCCLUSION_OFFSETS[32];
uniform int NUM_OCCLUSION_OFFSETS;
uniform float OCCLUSION_LIGHTING_RATIO;
uniform vec3 lightDirection;
uniform bool ambientOcclusionEnabled;
uniform ivec2 DATA_OFFSET;
uniform bool WRITE_DEBUG;

#include "common/constants.glsl"

float illumination(float dist, float relativeHeight) {
	return min(1.0, 1.0 - (atan(relativeHeight, dist) / (M_PI / 2)));
}

void main() {
	ivec2 dataPos = DATA_OFFSET + ivec2(gl_GlobalInvocationID.xy);

	vec4 heights = texelFetch(water0Read, dataPos);

	float occludedLightTotal = 0;
	float thisHeight = max(0.f, heights[2]);
	vec4 neighbourHeights = vec4(0);
	for (int i=0; i<4; i++) { 
		neighbourHeights[i] = max(0.f, texelFetch(water0Read, dataPos + OCCLUSION_OFFSETS[i])[2]);
	}
	vec2 slope = vec2(neighbourHeights[0] - neighbourHeights[2], neighbourHeights[1] - neighbourHeights[3]) / 2.f;

	if (ambientOcclusionEnabled) {
		for (int i=0; i<NUM_OCCLUSION_OFFSETS; i++) {
			ivec2 offset = OCCLUSION_OFFSETS[i];
			float dist = length(offset) * 16;
			float otherHeight;
			if (i < 4) {
				otherHeight = neighbourHeights[i];
			} else {
				otherHeight = max(0.f, texelFetch(water0Read, dataPos + offset)[2]);
			}
			vec2 slopeAdjust = (slope * offset);
			otherHeight -= (slopeAdjust.x + slopeAdjust.y);
			float relativeHeight = (otherHeight - thisHeight);
			occludedLightTotal += illumination(dist, relativeHeight);
		}
	}

	float occulusionLighting = 1.f;
	if (ambientOcclusionEnabled) {
		occulusionLighting = mix(1.f, clamp(occludedLightTotal / NUM_OCCLUSION_OFFSETS, 0.5f, 1.f), OCCLUSION_LIGHTING_RATIO);
	}

	vec4 output1 = vec4(occulusionLighting, 0.f, 0.f, 0.f);
	imageStore(lighting0Write, dataPos, output1);

    if (WRITE_DEBUG) {
        imageStore(debug0Write, dataPos, vec4(slope.x, slope.y, 0.f, 0.f));
        imageStore(debug1Write, dataPos, vec4(0.f));
    }
}