// Fragment shader for the first step of the GPU water flow simulation
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect rockProperties0Read;
layout (binding = 1) uniform sampler2DRect rockProperties1Read;
layout (binding = 2) uniform sampler2DRect water0Read;
layout (binding = 3) uniform sampler2DRect lava0Read;

layout (binding = 0, rgba8) uniform restrict image3D rockDepths0Write;
layout (binding = 1, rgba32f) uniform restrict image2D debug0Write;

#include "common/data/rocks.glsl"

uniform int Y_SIZE;
uniform int ROCK_DEPTH_RESOLUTION;
uniform bool WRITE_DEBUG;
uniform int DEBUG_LAYER;
uniform ivec2 DATA_OFFSET;

void main() {
	ivec2 dataPos = DATA_OFFSET + ivec2(gl_GlobalInvocationID.xy);

	vec4 rockProperties0Data = texelFetch(rockProperties0Read, dataPos);
	vec4 rockProperties1Data = texelFetch(rockProperties1Read, dataPos);
	vec4 water0Data = texelFetch(water0Read, dataPos);
	vec4 lava0Data = texelFetch(lava0Read, dataPos);
	RockLayers rockLayers = unpackRockLayers(rockProperties1Data);
	vec4 debugOutput = vec4(0);

	// find highest neighbour
	float neighbourHeights[8];
	ivec2 offsets[8];
	offsets[0] = ivec2(0, -1);
	offsets[1] = ivec2(1, 0);
	offsets[2] = ivec2(0, 1);
	offsets[3] = ivec2(-1, 0);
	offsets[4] = ivec2(-1, -1);
	offsets[5] = ivec2(1, -1);
	offsets[6] = ivec2(1, 1);
	offsets[7] = ivec2(-1, 1);

	for (int i=0; i<8; i++) {
		neighbourHeights[i] = texelFetch(water0Read, dataPos + offsets[i])[0];
	}

	int maxAt = 0, minAt = 0;
	float maxHeight = -99999.f;
	float minHeight = 99999.f;
	for (int i=0; i<8; i++) {
		if (neighbourHeights[i] > maxHeight) {
			maxAt = i;
			maxHeight = neighbourHeights[i];
		}
		if (neighbourHeights[i] < minHeight) {
			minAt = i;
			minHeight = neighbourHeights[i];
		}
	}

	// layers for highest neighbour
	RockLayer currentLayers[10];
	float neighbourLayerHeight = maxHeight;
	vec4 neighbourRockProperties0Data = texelFetch(rockProperties0Read, dataPos + offsets[maxAt]);
	vec4 minNeighbourRockProperties0Data = texelFetch(rockProperties0Read, dataPos + offsets[minAt]);
	vec4 neighbourRockProperties1Data = texelFetch(rockProperties1Read, dataPos + offsets[maxAt]);
	vec4 neighbourLava = texelFetch(lava0Read, dataPos + offsets[maxAt]);

	neighbourLayerHeight -= neighbourRockProperties0Data[1];
	if (neighbourLava[0] > 0.f) {
		currentLayers[0] = RockLayer(1, neighbourLayerHeight, true);
	} else {
		if (neighbourRockProperties0Data[1] > 1.f && neighbourLayerHeight - water0Data[0] < 8.f) {
			currentLayers[0] = RockLayer(0, neighbourLayerHeight, true);
		} else {
			if (neighbourLayerHeight - water0Data[0] < 64.f && neighbourRockProperties0Data[2] == rockProperties0Data[2]) {
				neighbourLayerHeight = min(neighbourLayerHeight, water0Data[0]);
			}

			currentLayers[0] = RockLayer(uint(neighbourRockProperties0Data[2]), neighbourLayerHeight, true);
		}
	}

	neighbourLayerHeight -= neighbourRockProperties0Data[3];
	currentLayers[1] = RockLayer(uint(neighbourRockProperties0Data[2]), neighbourLayerHeight, true);

	RockLayers neighbourRockLayers = unpackRockLayers(neighbourRockProperties1Data);

	for (int i=2; i<10; i++) {
		currentLayers[i] = neighbourRockLayers.layers[i - 2];
		neighbourLayerHeight -= currentLayers[i].depth;
		currentLayers[i].depth = neighbourLayerHeight;
	}

	// layers for here
	RockLayer layers[10];
	float layerHeight = water0Data[0];

	layerHeight -= rockProperties0Data[1];
	if (lava0Data[0] > 0.f) {
		layers[0] = RockLayer(1, layerHeight, true);
	} else {
		if (rockProperties0Data[1] > 1.f) {
			layers[0] = RockLayer(0, layerHeight, true);
		} else {
			if (layerHeight - minHeight < 64.f && minNeighbourRockProperties0Data[2] == rockProperties0Data[2]) {
				layerHeight = min(layerHeight, minHeight);
			}

			layers[0] = RockLayer(uint(rockProperties0Data[2]), layerHeight, true);
		}	
	}

	layerHeight -= rockProperties0Data[3];
	layers[1] = RockLayer(uint(rockProperties0Data[2]), layerHeight, true);

	for (int i=2; i<10; i++) {
		layers[i] = rockLayers.layers[i - 2];
		layerHeight -= layers[i].depth;
		layers[i].depth = layerHeight;
	}
	
	bool usingNeighbours = (maxHeight > water0Data[0]);
	if (!usingNeighbours) {
		currentLayers = layers;
	}

	int layerAt = 0;
	uint value = currentLayers[layerAt].type;

	vec4 amounts = vec4(0);
	bool first = true;
	bool changedLast = true;

	int yFrom = Y_SIZE - 1;
	int yTo = 0;
	yFrom = min(int(((maxHeight + max(lava0Data[0], neighbourLava[0]) + 64.f) / ROCK_DEPTH_RESOLUTION)), yFrom);
	yTo = max(int((min(layerHeight, minHeight) - 64.f) / ROCK_DEPTH_RESOLUTION), yTo);

	for (int y=yFrom; y >= 0; y--) {
		float height = y * ROCK_DEPTH_RESOLUTION;

		if (changedLast) {
			amounts = vec4(0);
			amounts[value] = 1.f;
			changedLast = false;
		}

		if (height < currentLayers[layerAt].depth && layerAt < 9) {
			layerAt++;
			float _mix = 1.f;
			amounts = vec4(0);
			if (value != currentLayers[layerAt].type) {
				_mix = (currentLayers[layerAt - 1].depth - height) / ROCK_DEPTH_RESOLUTION;
				amounts[value] = 1.f - _mix;

			}
			value = currentLayers[layerAt].type;
			amounts[value] = _mix;
			changedLast = true;
		}
		
		// if reading from neighbour layers and height < our layers top
		if (usingNeighbours && height < water0Data[0]) {
			currentLayers = layers;
			layerAt = 0;
			usingNeighbours = false;
		}

		first = false;

		value = currentLayers[layerAt].type;
		imageStore(rockDepths0Write, ivec3(dataPos.x, y, dataPos.y), amounts);

		if (WRITE_DEBUG) {
			imageStore(debug0Write, dataPos, debugOutput);	
		}
	}
}