// Fragment shader for the GPU water flow simulation pass that calculates start and length of autumn and winter seasons
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;

layout (binding = 0, rgba16) uniform restrict image2DRect temperature1Write;

uniform ivec2 water_size;

#include "common/temperature.glsl"
#include "common/temperature_mean.glsl"

float findFirstDayBelowZero(float landHeight, float meanTemperatureHereCelsius, float minCelsius, float maxCelsius) {
	if (minCelsius > 0.f) {
		return 2.f;
	}
	if (maxCelsius < 0.f) {
		return -1.f;
	}

	float minDay = DAY_OF_PEAK_TEMPERATURE, maxDay = (DAY_OF_PEAK_TEMPERATURE + 180) % 360;
	while (abs(minDay - maxDay) > 0.1f) {
		float middleDay = (minDay + maxDay) / 2.f;
		float temperature = meanTemperatureHereCelsius + getAddedTemperatureFromSeason(middleDay);
		if (temperature > 0) {
			minDay = middleDay;
		} else {
			maxDay = middleDay;
		}
	}

	return minDay / 360.f;
}

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);
	vec4 heights = texelFetch(water0Read, dataPos);

	float landHeight = max(0.f, heights[0]);
	float meanTemperatureHereCelsius = getMeanTemperature(landHeight, vec2(dataPos) / vec2(water_size), dataPos);
	float minCelsius = meanTemperatureHereCelsius - (TEMPERATURE_VARIATION_CELSIUS / 2.f);
	float maxCelsius = meanTemperatureHereCelsius + (TEMPERATURE_VARIATION_CELSIUS / 2.f);

	float lengthMultiplier = clamp(minCelsius / 5.f, 0.f, 1.f);
	float winterIntensity = 1.f - lengthMultiplier;
	if (lengthMultiplier > 0.8f && lengthMultiplier < 1.f) {
		lengthMultiplier = 0.8f;
	}
	float winterLength = mix(90.f, 0.f, lengthMultiplier);
	float winterStart = (360.f - winterLength) / 360.f;
	float autumnLength = mix(0.2f, 0.f, lengthMultiplier);
	float springStart = mix(45.f, 0.f, clamp((minCelsius + 15.f) / 20.f, 0.f, 1.f)) / 360.f;

	float firstDayWinter = min(winterStart, findFirstDayBelowZero(landHeight, meanTemperatureHereCelsius, minCelsius, maxCelsius) - (5.f / 360.f));
	imageStore(temperature1Write, dataPos, vec4(firstDayWinter - autumnLength, autumnLength, winterIntensity, springStart));
}