// Shader that steps the collision simulation
#version 150

uniform sampler2DRect data1; // position of collision points
uniform sampler2DRect data2; // velocity of collision points
uniform sampler2DRect data3; // acceleration of collision points
uniform ivec2 data1_size;
uniform int data1_resolution;
uniform float TIMESTEP;
uniform float CLOSENESS_FORCE;
uniform float CLOSENESS_RAMP_DISTANCE;
uniform float GRAVITY_FORCE;
uniform float VELOCITY_DAMP;
uniform float ANGLE_FORCE;
uniform float WALL_FORCE;
uniform float ALLOWED_CURVE;
uniform float TOUCHING_FORCE;
uniform float WALL_POSITION;
uniform vec3 ACCELERATION;

in vec2 texCoord;

out vec4 output1;
out vec4 output2;
out vec4 output3;

float clampClosenessForce(float multipler, float _active) {
	if (_active == 0) {
		return 0.0;
	}
	if (multipler > -CLOSENESS_RAMP_DISTANCE && multipler < CLOSENESS_RAMP_DISTANCE) {
		return multipler / CLOSENESS_RAMP_DISTANCE;
	} else {
		return clamp(multipler, -1.0, 1.0);
	}
}

vec3 getTouchingForce(vec3 position) {
	vec3 force = vec3(0);

	for (int x=0; x < data1_size.x; x++) {
		for (int z=0; z < data1_size.y; z++) {
			vec2 coords = vec2(0.5 + x, 0.5 + z);
			if (distance(coords, texCoord) > 1) {
				vec3 otherPosition = texture(data1, coords).xyz;
				float dist = distance(otherPosition, position);
				if (dist < 1) {
					force += normalize(position - otherPosition);
				}
			}
		}
	}
	return force;
}

vec3 getClosenessForce(vec3 position, vec3 velocity, vec4 neighbourPositions[4]) {
	vec3 topDirection = neighbourPositions[0].xyz - position;
	vec3 rightDirection = neighbourPositions[1].xyz - position;
	vec3 downDirection = neighbourPositions[2].xyz - position;
	vec3 leftDirection = neighbourPositions[3].xyz - position;

	float topMultiplier = clampClosenessForce(distance(topDirection, vec3(0)) - 1, neighbourPositions[0].a);
	vec3 topForce = topMultiplier != 0 ? topMultiplier * normalize(topDirection) : vec3(0);

	float rightMultiplier = clampClosenessForce(distance(rightDirection, vec3(0)) - 1, neighbourPositions[1].a);
	vec3 rightForce = rightMultiplier != 0 ? rightMultiplier * normalize(rightDirection) : vec3(0);

	float downMultiplier = clampClosenessForce(distance(downDirection, vec3(0)) - 1, neighbourPositions[2].a);
	vec3 downForce = downMultiplier != 0 ? downMultiplier * normalize(downDirection) : vec3(0);

	float leftMultiplier = clampClosenessForce(distance(leftDirection, vec3(0)) - 1, neighbourPositions[3].a);
	vec3 leftForce = leftMultiplier != 0 ? leftMultiplier * normalize(leftDirection) : vec3(0);

	vec3 force = leftForce + rightForce + topForce + downForce;
	// if (length(force) > 0 && length(velocity) > 0) {
	// 	float angle = acos(dot(normalize(force), normalize(velocity)));
	// 	return force * angle;
	// } else {
	// 	return force;
	// }
	return force;
}

vec3 getAngleForce(vec3 position, vec4 neighbourPositions[4]) {
	vec3 topPosition = neighbourPositions[0].xyz;
	vec3 rightPosition = neighbourPositions[1].xyz;
	vec3 downPosition = neighbourPositions[2].xyz;
	vec3 leftPosition = neighbourPositions[3].xyz;

	vec3 horizontalDirection = rightPosition - leftPosition;
	vec3 expectedHorizontalPosition = leftPosition + (horizontalDirection * 0.5);
	vec3 horizontalDelta = expectedHorizontalPosition - position;
	if (distance(horizontalDelta, vec3(0)) < ALLOWED_CURVE || neighbourPositions[3].w == 0 || neighbourPositions[1].w == 0) {
		horizontalDelta = vec3(0);
	}

	vec3 verticalDirection = downPosition - topPosition;
	vec3 expectedVerticalPosition = topPosition + (verticalDirection * 0.5);
	vec3 verticalDelta = expectedVerticalPosition - position;
	if (distance(verticalDelta, vec3(0)) < ALLOWED_CURVE || neighbourPositions[0].w == 0 || neighbourPositions[2].w == 0) {
		verticalDelta = vec3(0);
	}

	// horizontalDelta = vec3(0);
	// vec3 verticalDelta = vec3(0);

	return horizontalDelta + verticalDelta;
}

vec3 getGravityForce(vec3 position) {
	if (position.y < 0) {
		return vec3(0, 3, 0);
	}
	if (position.y > 0) {
		return vec3(0, -0.2, 0);
	}
	return vec3(0);
	// return vec3(0, -position.y, 0);
}

vec3 getAverageNeighbourVelocity(vec4 neighbourPositions[4], vec4 neighbourVelocities[4]) {
	float validNeighbours = (neighbourPositions[0].w + neighbourPositions[1].w + neighbourPositions[2].w + neighbourPositions[3].w);
	validNeighbours = max(validNeighbours, 1);
	vec3 average = ((neighbourPositions[0].w * neighbourVelocities[0].xyz) + 
			(neighbourPositions[1].w * neighbourVelocities[1].xyz) + 
			(neighbourPositions[2].w * neighbourVelocities[2].xyz) + 
			(neighbourPositions[3].w * neighbourVelocities[3].xyz)) / validNeighbours;
	return vec3(average.x, 0, average.z);
}

vec3 testWallForce(vec3 position) {
	if (position.z > WALL_POSITION) {
		return vec3(0, 0, -1);
	}
	return vec3(0, 0, 0);
}

void main() {
	vec4 firstRow = texture(data1, texCoord);
	vec4 secondRow = texture(data2, texCoord);
	vec4 thirdRow = texture(data3, texCoord);

	vec4 neighbourPositions[4];
	vec4 neighbourVelocities[4];

	neighbourPositions[0] = texture(data1, texCoord - vec2(0, 1));
	neighbourPositions[1] = texture(data1, texCoord + vec2(1, 0));
	neighbourPositions[2] = texture(data1, texCoord + vec2(0, 1));
	neighbourPositions[3] = texture(data1, texCoord - vec2(1, 0));

	neighbourVelocities[0] = texture(data2, texCoord - vec2(0, 1));
	neighbourVelocities[1] = texture(data2, texCoord + vec2(1, 0));
	neighbourVelocities[2] = texture(data2, texCoord + vec2(0, 1));
	neighbourVelocities[3] = texture(data2, texCoord - vec2(1, 0));

	if (firstRow.w == 1) {
		vec3 position = firstRow.xyz;
		vec3 velocity = secondRow.xyz;
		vec3 acceleration = thirdRow.xyz * TIMESTEP;

		vec3 velocityDamper = (velocity - (getAverageNeighbourVelocity(neighbourPositions, neighbourVelocities) * 0.95)) * VELOCITY_DAMP;
		// vec3 velocityDamper = velocity * VELOCITY_DAMP;
		velocity = (velocity - velocityDamper + acceleration) + 
			(getTouchingForce(position) * TIMESTEP * TOUCHING_FORCE) +
			(getClosenessForce(position, velocity, neighbourPositions) * TIMESTEP * CLOSENESS_FORCE) + 
			(getAngleForce(position, neighbourPositions) * TIMESTEP * ANGLE_FORCE) + 
			(getGravityForce(position) * TIMESTEP * GRAVITY_FORCE);

		position = position + (velocity * TIMESTEP);
		output1 = vec4(position, firstRow.w);
		// output1 = vec4(position.x, 2.06, position.z, firstRow.w);

		output2 = vec4(velocity , 0);
		output3 = thirdRow;
	} else {
		if (secondRow.w != -1) {
			vec2 copyCoords;
			vec2 offsetDirection;
			if (secondRow.w == 0) {
				offsetDirection = vec2(0, -1);
			}
			if (secondRow.w == 1) {
				offsetDirection =  vec2(1, 0);
			}
			if (secondRow.w == 2) {
				offsetDirection =  vec2(0, 1);
			}
			if (secondRow.w == 3) {
				offsetDirection =  vec2(-1, 0);
			}
			if (secondRow.w == 4) {
				offsetDirection = vec2(1, -1);
			}
			if (secondRow.w == 5) {
				offsetDirection =  vec2(1, 1);
			}
			if (secondRow.w == 6) {
				offsetDirection =  vec2(-1, 1);
			}
			if (secondRow.w == 7) {
				offsetDirection =  vec2(-1, -1);
			}
			copyCoords = texCoord + offsetDirection;

			vec3 copyPosition = texture(data1, copyCoords).xyz;
			output1 = vec4(copyPosition.x - (offsetDirection.x * 2), -4, copyPosition.z - (offsetDirection.y * 2), 0);
			output2 = secondRow;
			output3 = thirdRow;
		} else {
			output1 = firstRow;
			output2 = secondRow;
			output3 = thirdRow;
		}
	}
	//output1 = vec4(clamp(output1.x, -20.01, 20.01), output1.y, clamp(output1.z, -20.01, 20.01), output1.w);
}