// Fragment shader for the first step of the GPU lava flow simulation
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect lava0Read;
layout (binding = 1) uniform sampler2DRect lava1Read;
layout (binding = 2) uniform sampler2DRect lava2Read;
layout (binding = 3) uniform sampler2DRect lava3Read;
layout (binding = 4) uniform sampler2DRect water0Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect lava0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect lava1Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect lava2Write;
layout (binding = 3, rgba32f) uniform restrict image2DRect lava3Write;

uniform float MAX_FLOW_PERCENT_PER_STEP;
uniform float TIMESTEP;
uniform float FLOW_PERCENT;
uniform float FLOW_CUTOFF;
uniform float HEIGHT_DIFFERENTIAL_FOR_MAX_FLOW;
uniform float MIN_LAVA_REMAINING;
uniform float TEXTURE_FLOW_RATE;

vec4 getNeighbour(sampler2DRect image, ivec2 ourCoords, ivec2 offset) {
	return texelFetch(image, ourCoords + offset);
}

float getHeight(vec4 point) {
	return point[0] + point[1];
}

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	float totalFlow = 0.0;

	vec4 neighbours[4];
	ivec2 offsets[4];
	offsets[0] = ivec2(0, -1);
	offsets[1] = ivec2(1, 0);
	offsets[2] = ivec2(0, 1);
	offsets[3] = ivec2(-1, 0);
	neighbours[0] = vec4(getNeighbour(water0Read, dataPos, offsets[0])[0], getNeighbour(lava0Read, dataPos, offsets[0])[0], 0, 0);
	neighbours[1] = vec4(getNeighbour(water0Read, dataPos, offsets[1])[0], getNeighbour(lava0Read, dataPos, offsets[1])[0], 0, 0);
	neighbours[2] = vec4(getNeighbour(water0Read, dataPos, offsets[2])[0], getNeighbour(lava0Read, dataPos, offsets[2])[0], 0, 0);
	neighbours[3] = vec4(getNeighbour(water0Read, dataPos, offsets[3])[0], getNeighbour(lava0Read, dataPos, offsets[3])[0], 0, 0);

	vec4 lava = texelFetch(lava0Read, dataPos);
	vec4 flows = texelFetch(lava1Read, dataPos);
	vec4 display = texelFetch(lava2Read, dataPos);
	vec4 outputs = texelFetch(lava3Read, dataPos);
	outputs = vec4(outputs[0], 0, outputs[2], 0);
	vec4 land = texelFetch(water0Read, dataPos);

	float flowTemp = clamp((display[3] - 450.f) / 550.f, 0.05f, 1.f);
	float flowRate = MAX_FLOW_PERCENT_PER_STEP * flowTemp;
	float temperature = display[3];

	/* Allow lava to flow multiple directions at once */
	for (int i=0; i<4; i++) {
		float heightDifference = (land[0] + lava[0]) - getHeight(neighbours[i]);
		float lavaDifference = lava[0] - neighbours[i][1];
		float flow = (flowRate * ((heightDifference - MIN_LAVA_REMAINING) / HEIGHT_DIFFERENTIAL_FOR_MAX_FLOW));
		if (neighbours[i][1] > 0.f || lavaDifference > FLOW_CUTOFF) {
			flows[i] = max((flows[i] * FLOW_PERCENT) + flow, 0.f);
			totalFlow += flows[i];
		} else {
			flows[i] = 0.f;
		}
	}

	float totalFlowAvailable = (lava[0] - MIN_LAVA_REMAINING) * MAX_FLOW_PERCENT_PER_STEP;
	float flowScaling = 0.f;
	if (totalFlow > 0.f) {
		flowScaling =  clamp(totalFlowAvailable / totalFlow, 0.f, 1.f);
	}

	vec2 flowPercents = vec2(0);

	for (int i=0; i<4; i++) {
		flows[i] = flows[i] * flowScaling;
		outputs[1] -= flows[i];
		flowPercents -= offsets[i] * ((flows[i] / max(MIN_LAVA_REMAINING, lava[0] / 4.f)) * TEXTURE_FLOW_RATE * clamp(flowTemp * 1.f, 0.33f, 1.f));
	}

	display = display + vec4(flowPercents[0], flowPercents[1], 0, 0);

	imageStore(lava0Write, dataPos, lava);
	imageStore(lava1Write, dataPos, flows);
	imageStore(lava2Write, dataPos, display);
	imageStore(lava3Write, dataPos, outputs);
}