// Fragment shader for the second step of the GPU lava flow simulation
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect lava0Read;
layout (binding = 1) uniform sampler2DRect lava1Read;
layout (binding = 2) uniform sampler2DRect lava2Read;
layout (binding = 3) uniform sampler2DRect lava3Read;
layout (binding = 4) uniform sampler2DRect water0Read;
layout (binding = 5) uniform sampler2DRect rockProperties0Read;
layout (binding = 6) uniform sampler2DRect rockProperties1Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect lava0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect lava1Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect lava2Write;
layout (binding = 3, rgba32f) uniform restrict image2DRect lava3Write;
layout (binding = 4, rgba32f) uniform restrict image2DRect water0Write;
layout (binding = 5, rgba32f) uniform restrict image2DRect rockProperties0Write;
layout (binding = 6, rgba32f) uniform restrict image2DRect rockProperties1Write;

uniform float WATER_IMPACT;
uniform float TIMESTEP;

#include "common/data/rocks.glsl"

vec4 getNeighbour(sampler2DRect image, ivec2 ourCoords, ivec2 offset) {
	return texelFetch(image, ourCoords + offset);
}

int oppositeDirection(int direction) {
	if (direction == 0 || direction == 2) { 
		return 2 - direction; 
	} else { 
		return 4 - direction; 
	};
}

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	ivec2 offsets[4];
	offsets[0] = ivec2(0, -1);
	offsets[1] = ivec2(1, 0);
	offsets[2] = ivec2(0, 1);
	offsets[3] = ivec2(-1, 0);

	vec4 neighbourLava[8];
	neighbourLava[0] = getNeighbour(lava2Read, dataPos, ivec2(0, -1));
	neighbourLava[1] = getNeighbour(lava2Read, dataPos, ivec2(1, 0));
	neighbourLava[2] = getNeighbour(lava2Read, dataPos, ivec2(0, 1));
	neighbourLava[3] = getNeighbour(lava2Read, dataPos, ivec2(-1, 0));
	neighbourLava[4] = getNeighbour(lava2Read, dataPos, ivec2(-1, -1));
	neighbourLava[5] = getNeighbour(lava2Read, dataPos, ivec2(1, 1));
	neighbourLava[6] = getNeighbour(lava2Read, dataPos, ivec2(1, -1));
	neighbourLava[7] = getNeighbour(lava2Read, dataPos, ivec2(-1, 1));

	vec4 neighbourFlows[4];
	neighbourFlows[0] = getNeighbour(lava1Read, dataPos, ivec2(0, -1));
	neighbourFlows[1] = getNeighbour(lava1Read, dataPos, ivec2(1, 0));
	neighbourFlows[2] = getNeighbour(lava1Read, dataPos, ivec2(0, 1));
	neighbourFlows[3] = getNeighbour(lava1Read, dataPos, ivec2(-1, 0));

	vec4 lava = texelFetch(lava0Read, dataPos);
	vec4 flows = texelFetch(lava1Read, dataPos);
	vec4 display = texelFetch(lava2Read, dataPos);
	vec4 outputs = texelFetch(lava3Read, dataPos);
	vec4 land = texelFetch(water0Read, dataPos);
	vec4 rockProperties0Data = texelFetch(rockProperties0Read, dataPos);
	vec4 rockProperties1Data = texelFetch(rockProperties1Read, dataPos);

	float flowRate = 1.f;
	// float addRate = (sin((step_count + (outputs[2] * 512.f * 2.f * 3.1415926)) / 512.f) + 1.f) / 2.f;
	float addRate = 1.f;
	
	/* A temperature of 0 represents the ambient temperature */
	float temperature = display[3];
	/* TemperatureQuantity represents total energy stored, as opposed to temperature which is energy per unit */
	float temperatureQuantity = max(temperature * lava[0], 0.f);
	float temperatureUnits = max(lava[0], 0.01f);

	if (TIMESTEP > 0.f) {
		/* Temperature loss due to ambient cooling */
		float ambientTemperatureAmount = lava[0] * clamp((0.5f + ((8.f / max(lava[0], 0.01f)) * 0.5f)), 0.f, 1.f);
		temperatureQuantity -= ((temperature * ambientTemperatureAmount) / 1000.f) * 1.f;

		vec2 idealTextureOffset = vec2(0);
		int idealCount = 0;
		float lavaToConvertToLand = 0.f;
		bool differentTemperatureFlow = (outputs[0] > 0.f && temperature < 800.f);

		/* Account for temperature changing from incoming flows */
		for (int i=0; i<4; i++) {
			int opposite = oppositeDirection(i);
			if (neighbourFlows[i][opposite] > 0.f) {
				if ((neighbourLava[i][3] - temperature) > 100.f) {
					differentTemperatureFlow = true;
				}
				if (lava[0] <= 0.f) {
					display[0] = neighbourLava[i][0];
					display[1] = neighbourLava[i][1];
				}
			}
			outputs[1] += neighbourFlows[i][opposite];
			temperatureQuantity += (neighbourLava[i][3] * neighbourFlows[i][opposite]);
		}

		if (differentTemperatureFlow) {
			float amt = max(0.f, 0.2f - min(outputs[1], 0.f));
			lavaToConvertToLand = max(0.f, lava[0] - amt);
			temperatureQuantity -= (lavaToConvertToLand * temperature);
			temperatureUnits -= lavaToConvertToLand;
		}

		/* Account for temperature changing from outgoing flows */
		for (int i=0; i<4; i++) {
			temperatureQuantity -= (temperature * flows[i]);
		}

		/* Account for temperature changing from generating lava */
		temperatureQuantity += (outputs[0] * 1000.f * addRate);

		/* Temperature changes from water */
		float waterHeight = max(land[1], -land[0]);
		if (waterHeight > 0.02) {
			temperatureUnits += (waterHeight * WATER_IMPACT);
		}

		float addedLava = min((outputs[0] * addRate) + (outputs[1] * flowRate), 4096.f - land[2]);
		lava[0] += addedLava;
		lava[0] = max(0.0, lava[0]);
		lava[1] = addedLava; // simply set for debugging- this could be removed
		temperatureUnits += addedLava;
		outputs[0] = 0.0;
		outputs[1] = 0.0;

		display[3] = clamp(temperatureQuantity / temperatureUnits, 0.f, 1000.f);
		// if (display[3] < 400.f && !differentTemperatureFlow) {
		// 	lavaToConvertToLand = ((400.f - lava[0]) / 200.f) * 0.01f * max(0.f, lava[0] - MIN_LAVA_REMAINING);
		// }
		if (display[3] < 200.f && !differentTemperatureFlow) {
			lavaToConvertToLand = lava[0];
		}

		/* Convert lava to land */
		lavaToConvertToLand = min(lavaToConvertToLand, lava[0]);
		lava[3] = lavaToConvertToLand;
		lava[0] = max(0.f, lava[0] - lavaToConvertToLand);
		land[0] += lavaToConvertToLand;

		/* Adjust rocks for lava that was converted */
		if (lavaToConvertToLand > 0.f) {
			RockLayers rockLayers = unpackRockLayers(rockProperties1Data);
			if (uint(rockProperties0Data[2]) != 1) {
				pushDownRockLayers(rockLayers);
				rockLayers.layers[0].type = uint(rockProperties0Data[2]);
				rockLayers.layers[0].depth = rockProperties0Data[3];
				rockProperties0Data[2] = 1;
				rockProperties0Data[3] = 0.f;
			}
			if (rockProperties0Data[1] > 16.f) {
				pushDownRockLayers(rockLayers);
				rockLayers.layers[0].type = 0;
				rockLayers.layers[0].depth = rockProperties0Data[1];
				rockProperties0Data[2] = 1;
				rockProperties0Data[3] = 0.f;
			}
			rockProperties1Data = packRockLayers(rockLayers);
			rockProperties0Data[1] = 0.f;
			rockProperties0Data[3] += lavaToConvertToLand;
		}
	} else {
		temperatureQuantity += (outputs[0] * 1000.f * addRate);
		float addedLava = min(outputs[0] * addRate, 4096.f - land[2]);
		lava[0] += addedLava;
		lava[0] = max(0.0, lava[0]);
		temperatureUnits += addedLava;
		outputs[0] = 0.0;
		outputs[1] = 0.0;
		display[3] = clamp(temperatureQuantity / temperatureUnits, 0.f, 1000.f);
	}

	/* Compute attributes needed to display lava */
 	if (lava[0] <= 0.f) {
 		if (neighbourLava[0][2] <= -1.f && neighbourLava[1][2] <= -1.f && neighbourLava[2][2] <= -1.f && neighbourLava[3][2] <= -1.f &&
 			neighbourLava[4][2] <= -1.f && neighbourLava[5][2] <= -1.f && neighbourLava[6][2] <= -1.f && neighbourLava[7][2] <= -1.f) {
 			display[2] = -3.f;
 		} else {
 			temperatureQuantity = 0.f;
 			for (int i=0; i<8; i++) {
 				if (neighbourLava[i][2] > 0 && neighbourLava[i][3] > temperatureQuantity) {
 					temperatureQuantity = neighbourLava[i][3];
 				}
 			}
 			temperatureUnits = 1;
 			display[2] = -1.f;
 		}
 		display[0] = 0.f;
 		display[1] = 0.f;
 	} else {
 		display[2] = lava[0];
 	}

	imageStore(lava0Write, dataPos, lava);
	imageStore(lava1Write, dataPos, flows);
	imageStore(lava2Write, dataPos, display);
	imageStore(lava3Write, dataPos, outputs);
	imageStore(water0Write, dataPos, land);
	imageStore(rockProperties0Write, dataPos, rockProperties0Data);
	imageStore(rockProperties1Write, dataPos, rockProperties1Data);
}