// Fragment shader for the GPU simulation that generates animal coverage information
#version 150

uniform sampler2DRect data1; // water row 0
uniform sampler2D data2; // coverages row 0
uniform sampler2DRect data3; // lava row 0
uniform int data1_resolution;
uniform ivec2 data1_offset;
uniform int data2_resolution;
uniform ivec2 data2_offset;
uniform int data3_resolution;
uniform ivec2 data3_offset;
uniform int output_resolution;
uniform ivec2 output_offset;

in vec2 texCoord;

out uvec4 output1; // animals row 0

highp float rand(vec2 co) {

	int x = int(co.x);
	int y = int(co.y);
	highp float result = 0.f;

	for (int i = 0; i < 4; i++) {
		x += (x << 29);
		x ^= (x << 11);
		x ^= (y << 17);
		x ^= (x >> 19);
		x ^= (y >> 6);
		x ^= (x << 4);
	}

	result = x;
	
	return (x & 8191) / 8192.0;
}

float scale(int fromResolution, int toResolution) {
	int numerator = 1, denominator = 1;
	if (fromResolution > 0) {
		numerator *= fromResolution;
	} else {
		denominator *= -fromResolution;
	}
	if (toResolution > 0) {
		denominator *= toResolution;
	} else {
		numerator *= -toResolution;
	}
	return numerator / float(denominator);
}

vec2 convertCoordinates(vec2 coordinates, int inputResolution, int outputResolution, vec2 inputOffset, vec2 outputOffset) {
	float _scale = scale(inputResolution, outputResolution);
	return vec2((coordinates.x * _scale) + (inputOffset.x * _scale) - (outputOffset.x),
				(coordinates.y * _scale) + (inputOffset.y * _scale) - (outputOffset.y));
}

uint calculateDeerDensity(vec4 water, vec4 coverages) {
	float waterFactor = (water.x < 0 || water.y > 1) ? 0.f : 1.f;
	float treesFactor = 1.f - (coverages[0] * 5.f);
	float swampFactor = (0.5f - coverages[1]) * 2.f;
	float grassFactor = coverages[2];
	return uint(clamp(waterFactor * treesFactor * swampFactor * grassFactor, 0.f, 1.f) * 255);
}

uint calculateCrabDensity(vec4 water, vec4 lava, vec4 coverages) {
	float heightFactor = clamp((2.0 - abs(water.x)) / 2.0, 0.f, 1.f);
	float lavaFactor = lava[0] == 0.f ? 1.f : 0.f;
	return uint(heightFactor * lavaFactor * 255);
}

uint calculateIbexDensity(vec4 water, vec4 coverages) {
	float treeCoverageCutoff = 0.15f;
	float treesFactor = clamp(treeCoverageCutoff - coverages[0], 0.f, treeCoverageCutoff) * (1.f / treeCoverageCutoff);
	float heightFactor = clamp((water.r - 1024.f) / 1024.f, 0.f, 1.f);
	float grassFactor = coverages[2];
	return uint(treesFactor * heightFactor * grassFactor * 255);
}

uint calculateCrocDensity(vec4 water, vec4 coverages4x4) {
	const float cutoff = 0.75;
	float swampFactor = clamp((coverages4x4[1] - cutoff) * (1.f / (1.f - cutoff)), 0.f, 1.f);
	float heightFactor = clamp((128 - water.r) / 128.f, 0.f, 1.f);
	return uint(swampFactor * heightFactor * 255);
}

uint calculateRabbitDensity(vec4 water, vec4 coverages) {
	const float idealForestCoverage = 0.8f;
	const float leeway = 0.25f;
	float treesFactor = clamp(1.f - (abs(idealForestCoverage - coverages[0]) / leeway), 0.f, 1.f);
	float grassFactor = coverages[2];
	return uint(treesFactor * grassFactor * 255);
}

uint calculateBearDensity(vec4 water, vec4 coverages) {
	const float cutoff = 0.75f;
	float forestFactor = clamp((coverages.x - cutoff) * (1.f / (1.f - cutoff)), 0.f, 1.f);
	return uint(forestFactor * 255);
}

void main() {
	vec2 waterPosition = convertCoordinates(texCoord.xy - vec2(0.5, 0.5), output_resolution, data1_resolution, output_offset, data1_offset) + vec2(0.5, 0.5);
	vec2 coveragesPosition = convertCoordinates(texCoord.xy - vec2(0.5, 0.5), output_resolution, data3_resolution, output_offset, data3_offset) + vec2(0.5, 0.5);
	vec4 water = texture(data1, waterPosition);
	vec4 lava = texture(data3, waterPosition);
	vec4 coverages = texelFetch(data2, ivec2(waterPosition - vec2(0.5, 0.5)), 0);
	vec4 coverages4x4 = texelFetch(data2, ivec2(waterPosition - vec2(0.5, 0.5)) / 4, 2);

	uint deerCoverage = calculateDeerDensity(water, coverages);
	uint crabCoverage = calculateCrabDensity(water, lava, coverages) * uint(256);
	uint ibexDensity = calculateIbexDensity(water, coverages) * uint(256 * 256);
	uint crocDensity = calculateCrocDensity(water, coverages4x4) * uint(256 * 256 * 256);
	uint rabbitDensity = calculateRabbitDensity(water, coverages);
	uint bearDensity = calculateBearDensity(water, coverages) * uint(256);
	output1 = uvec4(deerCoverage | crabCoverage | ibexDensity | crocDensity, rabbitDensity | bearDensity, 0, 0);
}