struct GrassData {
	uint colour;
	uint growth;
	uint savedGrowth;
	uint savedGrowthCounter;
};

GrassData unpackGrassData(inout uint value) {
	return GrassData(value >> 24, (value & 0x00ff0000) >> 16, (value & 0x0000ff00) >> 8, (value & 0x000000ff));
}

uint packGrassData(inout GrassData value) {
	return (value.colour << 24) | (value.growth << 16) | (value.savedGrowth << 8) | (value.savedGrowthCounter);
}

vec4 updateGrass(inout EnvironmentInfo info) {
	GrassData data = unpackGrassData(info.additionalData[0][0]);
	if (dayProgressed) {
		uint colourToday = uint(255.f * clamp((info.groundwaterMM - 150) / 350.f, 0.f, 1.f));

		float groundwaterRangeMM = info.groundwaterData.maxGroundwaterLastYearMM - info.groundwaterData.minGroundwaterLastYearMM;
		if (groundwaterRangeMM > 0.f) {
			float totalPossibleRange = 1.f - min((info.groundwaterData.maxGroundwaterLastYearMM / info.groundwaterData.minGroundwaterLastYearMM) / 5.f, 1.f);
			colourToday = min(colourToday,
				uint(clamp(((info.groundwaterMM - info.groundwaterData.minGroundwaterLastYearMM) / groundwaterRangeMM) + totalPossibleRange, 0.f, 1.f) * 255.f)
			);
		}

		if (colourToday > data.colour) {
			data.colour = clamp(data.colour + 8u, 0u, colourToday);
		}
		if (data.colour > 0) {
			data.colour -= 1u;
		}

		float groundWaterFactor = clamp((info.groundwaterMM - 250) / 250.f, 0.f, 1.f);
		float inundationFactor = range(0, 0, info.inundationData.daysInnundatedLastYear, 190, 210);
		uint coverageNow = uint(inundationFactor * groundWaterFactor * info.normalFactor * 255u);
		if (info.dirtDepth < 1.f || info.iceDepth > 0.f || info.inLava || info.waterData.landHeight < 2.f) {
			data.growth = 0u;
			coverageNow = 0u;
			data.savedGrowthCounter = 0u;
		}
		if (coverageNow > data.growth) {
			coverageNow = min(coverageNow, data.growth + 8u);
		}
		if (coverageNow >= data.savedGrowth || data.savedGrowthCounter == 0u) {
			data.savedGrowth = coverageNow;
			data.savedGrowthCounter = 12u;
		}
		if (data.savedGrowthCounter > 0u) {
			coverageNow = max(coverageNow, data.savedGrowth);
		}
		if (coverageNow < data.growth) {
			data.growth = data.growth - 1u;
		} else {
			data.growth = coverageNow;
		}
	}
	if (monthProgressed && data.savedGrowthCounter > 0u) {
		data.savedGrowthCounter -= 1;
	}
	info.additionalData[0][0] = packGrassData(data);
	
	float colour = 0.90625f - ((data.colour / 255.f) / 2.f);
	float amount = data.growth / 255.f;
	if (info.temperature.minCelsius < 5.f) {
		float winterProgress = info.autumnProgress;
		if (info.winterProgress > 0.f) {
			winterProgress = 1.f;
		}
		if (winterProgress > 0.f) {
			colour = max(colour, 0.5f + (winterProgress * 0.40625f * info.winterIntensity));
		}
		if (info.springProgress < 1.f) {
			colour = max(colour, 0.5f + ((1.f - min(info.springProgress * 2.f, 1.f)) * 0.40625f * info.winterIntensity));
		}
	}

	return vec4(amount, colour, 0.f /* snow coverage, set elsewhere */, 0.f /* marsh coverage, set elsewhere */);
}