// Compute shader for the GPU simulation that updates biomes
#version 440

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;
layout (binding = 1) uniform sampler2DRect water2Read;
layout (binding = 2) uniform sampler2DRect rockProperties0Read;
layout (binding = 3) uniform usampler2DRect additional0Read;
layout (binding = 4) uniform usampler2DRect additional1Read;
layout (binding = 5) uniform usampler2DRect additional2Read;
layout (binding = 6) uniform usampler2DRect additional3Read;
layout (binding = 7) uniform sampler2DRect rainfall2Read;
layout (binding = 8) uniform sampler2DRect lava0Read;
layout (binding = 9) uniform sampler2DRect temperature0Read;
layout (binding = 10) uniform sampler2DRect temperature1Read;
layout (binding = 11) uniform sampler2DRect ice1Read;
layout (binding = 12) uniform sampler2DRect ice2Read;
layout (binding = 13) uniform sampler2DRect water1Read; //unused
layout (binding = 14) uniform sampler2DRect water3Read; //unused

layout (binding = 0, rgba8) uniform writeonly restrict image2DRect biome0Write;
layout (binding = 1, rgba8) uniform writeonly restrict image2DRect biome1Write;
layout (binding = 2, rgba8) uniform writeonly restrict image2DRect biome2Write;
layout (binding = 3, rgba32ui) uniform writeonly restrict uimage2DRect additional0Write;
layout (binding = 4, rgba32ui) uniform writeonly restrict uimage2DRect additional1Write;
layout (binding = 5, rgba32ui) uniform writeonly restrict uimage2DRect additional2Write;
layout (binding = 6, rgba32ui) uniform writeonly restrict uimage2DRect additional3Write;
layout (binding = 7, rgba32f) uniform writeonly restrict image2DRect debugOutput0Write;

#include "common/temperature.glsl"
#include "common/data/biomes.glsl"
#include "common/noise.glsl"
#include "common/resolutions.glsl"
#include "common/data/water.glsl"
#include "common/util.glsl"

uniform bool dayProgressed;
uniform bool monthProgressed;
uniform bool yearProgressed;
uniform int stepCount;
uniform float RAINFALL_NORMALISED_TO_SIM_CONVERSION;
uniform float RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
uniform float RIVER_THRESHOLD;
uniform vec4 randomNumbers;
uniform int water_resolution;
uniform bool WRITE_DEBUG;
uniform float DAY_OF_YEAR;

struct PresentBiome {
	vec4 biomeData;
	float targetCoverage;
};

void pushBiomeAmount(vec4 biomeData, float targetCoverage, inout PresentBiome majorBiome, inout PresentBiome minorBiome) {
	if (targetCoverage > majorBiome.targetCoverage) {
		minorBiome = majorBiome;
		majorBiome.biomeData = biomeData;
		majorBiome.targetCoverage = targetCoverage;
	} else {
		if (targetCoverage > minorBiome.targetCoverage) {
			minorBiome.biomeData = biomeData;
			minorBiome.targetCoverage = targetCoverage;
		}
	}
}

struct InundationData {
	uint daysInnundatedLastYear;
	uint daysInnundatedThisYear;
	uint unused1;
	uint unused2;
};

struct GrowthData {
	uint coverage;
	uint growthTimer;
	uint targetCoverage;
	uint currentYearMinimumCoverage;
};

struct GroundwaterData {
	float maxGroundwaterLastYear;
	float minGroundwaterLastYear;
	float maxGroundwaterThisYear;
	float minGroundwaterThisYear;
	float maxGroundwaterLastYearMM;
	float minGroundwaterLastYearMM;
};

GrowthData unpackGrowthData(inout uint value) {
	return GrowthData(value >> 24, (value & 0x00ff0000) >> 16, (value & 0x0000ff00) >> 8, (value & 0x000000ff));
}

uint packGrowthData(inout GrowthData value) {
	return (value.coverage << 24) | (value.growthTimer << 16) | (value.targetCoverage << 8) | (value.currentYearMinimumCoverage);
}

GroundwaterData unpackGroundwaterData(inout uint value) {
	GroundwaterData data = GroundwaterData(
		(value >> 24) / 255.f, ((value & 0x00ff0000) >> 16) / 255.f, ((value & 0x0000ff00) >> 8) / 255.f, (value & 0x000000ff) / 255.f,
		0.f, 0.f);
	data.maxGroundwaterLastYearMM = data.maxGroundwaterLastYear * RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
	data.minGroundwaterLastYearMM = data.minGroundwaterLastYear * RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
	return data;
}

uint packGroundwaterData(inout GroundwaterData value) {
	return (uint(value.maxGroundwaterLastYear * 255u) << 24) | (uint(value.minGroundwaterLastYear * 255u) << 16) | 
		(uint(value.maxGroundwaterThisYear * 255u) << 8) | uint(value.minGroundwaterThisYear * 255u);
}

struct EnvironmentInfo {
	float rainfallMM;
	float yearlyRainfallMM;
	float groundwaterMM;
	WaterData waterData;
	vec4 rockPropertiesData;
	float dirtDepth;
	InundationData inundationData;
	uvec4 additionalData[4];
	bool canGrow;
	float normalFactor;
	Temperature temperature;
	float iceDepth;
	float snowCoverage;
	float autumnProgress, winterProgress, springProgress, winterIntensity;
	bool inLava;
	bool standardExclusions;
	GroundwaterData groundwaterData;
};

#include "operations/plants/biomes/grass.glsl"
#include "operations/plants/biomes/marsh.glsl"
#include "operations/plants/biomes/swamp.glsl"
#include "operations/plants/biomes/riparian.glsl"
#include "operations/plants/biomes/temperate_forest.glsl"
#include "operations/plants/biomes/tropical_rainforest.glsl"
#include "operations/plants/biomes/oasis.glsl"
#include "operations/plants/biomes/savannah.glsl"
#include "operations/plants/biomes/mangrove_swamp.glsl"
#include "operations/plants/biomes/boreal_forest.glsl"
#include "operations/plants/biomes/hot_desert.glsl"
#include "operations/plants/biomes/inundation.glsl"
#include "operations/plants/biomes/rock.glsl"

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	PresentBiome majorBiome, minorBiome;
	majorBiome.biomeData = vec4(15.f / 255.f, 0.f, 0.f, 0.f);
	majorBiome.targetCoverage = 0.00001f;
	minorBiome.biomeData = vec4(15.f / 255.f, 0.f, 0.f, 0.f);
	minorBiome.targetCoverage = 0.f;

	EnvironmentInfo info;
	info.additionalData[0] = texelFetch(additional0Read, dataPos);
	info.additionalData[1] = texelFetch(additional1Read, dataPos);
	info.additionalData[2] = texelFetch(additional2Read, dataPos);
	info.additionalData[3] = texelFetch(additional3Read, dataPos);
	vec4 lava0Data = texelFetch(lava0Read, dataPos);
	info.waterData = getWaterData(dataPos, false, true, false);
	info.rockPropertiesData = texelFetch(rockProperties0Read, dataPos);
	info.dirtDepth = info.rockPropertiesData[1];
	info.yearlyRainfallMM = texelFetch(rainfall2Read, dataPos)[0] * RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
	float rainfallNormalised = info.waterData.rainfall / RAINFALL_NORMALISED_TO_SIM_CONVERSION;
	info.rainfallMM = rainfallNormalised * RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
	info.canGrow = true;
	info.groundwaterData = unpackGroundwaterData(info.additionalData[2][3]);
	info.groundwaterData.maxGroundwaterThisYear = max(info.groundwaterData.maxGroundwaterThisYear, info.waterData.groundWater);
	info.groundwaterData.minGroundwaterThisYear = min(info.groundwaterData.minGroundwaterThisYear, info.waterData.groundWater);
	info.groundwaterData.maxGroundwaterLastYear = max(info.groundwaterData.maxGroundwaterLastYear, info.waterData.groundWater);
	info.groundwaterData.minGroundwaterLastYear = min(info.groundwaterData.minGroundwaterLastYear, info.waterData.groundWater);
	if (yearProgressed) {
		info.groundwaterData.maxGroundwaterLastYear = info.groundwaterData.maxGroundwaterThisYear;
		info.groundwaterData.minGroundwaterLastYear = info.groundwaterData.minGroundwaterThisYear;
		info.groundwaterData.maxGroundwaterThisYear = info.waterData.groundWater;
		info.groundwaterData.minGroundwaterThisYear = info.waterData.groundWater;
	}
	info.additionalData[2][3] = packGroundwaterData(info.groundwaterData);
	info.groundwaterMM = info.waterData.groundWater * RAINFALL_NORMALISED_TO_DISPLAY_MM_COVERSION;
	info.inundationData = updateInundation(info);
	info.temperature = normalisedToTemperature(texelFetch(temperature0Read, dataPos).xy);
	vec4 temperatureSeasons = texelFetch(temperature1Read, dataPos);
	info.iceDepth = texelFetch(ice1Read, dataPos)[0];
	info.snowCoverage = texelFetch(ice2Read, dataPos)[0];
	info.inLava = lava0Data[0] > 0.f;
	info.standardExclusions = (info.iceDepth > 0.f || info.inLava || info.dirtDepth < 1.f || info.waterData.waterHeight > 0.02f || info.waterData.landHeight < 2.f);

	info.autumnProgress = 0.f;
	info.winterProgress = 0.f;
	info.springProgress = 1.f;
	info.winterIntensity = temperatureSeasons[2];
	float yearProgress = DAY_OF_YEAR / 360.f;
	float winterStart = temperatureSeasons[0] + temperatureSeasons[1];

	if (yearProgress > winterStart) {
		float winterLength = 1.f - winterStart;
		info.winterProgress = clamp((yearProgress - winterStart) / winterLength, 0.f, 1.f);
	} else {
		if (yearProgress > temperatureSeasons[0]) {
			info.autumnProgress = clamp((yearProgress - temperatureSeasons[0]) / temperatureSeasons[1], 0.f, 1.f);
		} else {
			info.springProgress = min((yearProgress - temperatureSeasons[3]) * 4.f, 1.f);
			if (info.springProgress < 0.f) {
				info.springProgress = 0.f;
				info.winterProgress = 1.f;
			}
		}
	}

	vec3 normal = getNormalFromNeighbours(vec3(0, info.waterData.totalHeight, 0), 
		vec3(0, texelFetch(water0Read, dataPos + ivec2(0, -1))[2], -water_resolution),
		vec3(water_resolution, texelFetch(water0Read, dataPos + ivec2(1, 0))[2], 0),
		vec3(0, texelFetch(water0Read, dataPos + ivec2(0, 1))[2], water_resolution),
		vec3(-water_resolution, texelFetch(water0Read, dataPos + ivec2(-1, 0))[2], 0)
	);
	info.normalFactor = 1.f;
	if (normal.y < 0.6) {
		info.normalFactor = clamp((normal.y - 0.5f) / 0.1f, 0.f, 1.f);
	}

	vec4 grass = updateGrass(info);
	grass[2] = info.snowCoverage;

	float targetCoverage = 0.f;
	float forestCoverage = 0.f;

	vec4 riparian = updateRiparian(info, targetCoverage);
	if (targetCoverage > 0.f) {
		pushBiomeAmount(riparian, targetCoverage, majorBiome, minorBiome);
		grass[0] = grass[0] - riparian[1];
	}

	vec4 swamp = updateSwamp(info, targetCoverage);
	if (targetCoverage > 0.f) {
		pushBiomeAmount(swamp, targetCoverage, majorBiome, minorBiome);
		grass[0] = max(0.f, grass[0] - swamp[1]);
	}

	vec4 marsh = updateMarsh(info, targetCoverage);
	if (targetCoverage > 0.f) {
		pushBiomeAmount(marsh, targetCoverage, majorBiome, minorBiome);
	}
	grass[3] = marsh[1];

	float tropicalRainforestCoverage = 0.f;
	vec4 tropicalRainforest = updateTropicalRainforest(info, tropicalRainforestCoverage);
	if (tropicalRainforestCoverage > 0.f) {
		pushBiomeAmount(tropicalRainforest, tropicalRainforestCoverage, majorBiome, minorBiome);
		forestCoverage += tropicalRainforest[1];
	}

	vec4 temperateForest = updateTemperateForest(info, targetCoverage, tropicalRainforestCoverage);
	if (targetCoverage > 0.f) {
		targetCoverage = min(targetCoverage, 1.f - tropicalRainforestCoverage);
		pushBiomeAmount(temperateForest, targetCoverage, majorBiome, minorBiome);
		forestCoverage += temperateForest[1];
	}

	vec4 borealForest = updateBorealForest(info, targetCoverage);
	if (targetCoverage > 0.f) {
		targetCoverage = min(targetCoverage, 1.f - tropicalRainforestCoverage);
		pushBiomeAmount(borealForest, targetCoverage, majorBiome, minorBiome);
		forestCoverage += borealForest[1];
	}

	float savannahCoverage = 0.f;
	vec4 savannah = updateSavannah(info, savannahCoverage, forestCoverage);
	if (savannahCoverage > 0.f) {
		pushBiomeAmount(savannah, savannahCoverage, majorBiome, minorBiome);
	}

	vec4 oasis = updateOasis(info, targetCoverage, forestCoverage + savannah[1]);
	if (targetCoverage > 0.f) {
		pushBiomeAmount(oasis, targetCoverage, majorBiome, minorBiome);
	}

	vec4 mangroveSwamp = updateMangroveSwamp(info, targetCoverage);
	if (mangroveSwamp[1] > 0.f) {
		pushBiomeAmount(mangroveSwamp, targetCoverage, majorBiome, minorBiome);
		forestCoverage += mangroveSwamp[1];
	}

	vec4 hotDesert = updateHotDesert(info, targetCoverage);
	if (targetCoverage > 0.f) {
		pushBiomeAmount(hotDesert, targetCoverage, majorBiome, minorBiome);
	}

	float biomeTotalCoverage = majorBiome.targetCoverage + minorBiome.targetCoverage;
	if (biomeTotalCoverage > 1.f) {
		majorBiome.biomeData[1] /= biomeTotalCoverage;
		minorBiome.biomeData[1] /= biomeTotalCoverage;
		forestCoverage /= biomeTotalCoverage;
	}

	grass[0] = grass[0] - (clamp((forestCoverage - 0.33f) * 2.f, 0.f, 1.f));
	grass[0] = clamp(grass[0], 0.f, 1.f);

	if (biomeTotalCoverage < 1.f) {
		float grasslandsBiomes = min(grass[0], 1.f - biomeTotalCoverage);
		biomeTotalCoverage += grasslandsBiomes;
		pushBiomeAmount(vec4(4.f / 255.f, grasslandsBiomes, 0.f, 0.f), grasslandsBiomes, majorBiome, minorBiome);
	}

	if (info.snowCoverage > 0.f) {
		grass[0] = 0.f;
		grass[1] = 1.2f;
	}

	vec4 rockBiome = updateRock(info);
	if (biomeTotalCoverage < 1.f && rockBiome[1] > 0.f) {
		float rockCoverage = min(rockBiome[1], 1.f - biomeTotalCoverage);
		pushBiomeAmount(rockBiome, rockCoverage, majorBiome, minorBiome);
	}

	biomeTotalCoverage = majorBiome.biomeData[1] + minorBiome.biomeData[1];
	if (biomeTotalCoverage > 1.f) {
		majorBiome.biomeData[1] /= biomeTotalCoverage;
		minorBiome.biomeData[1] /= biomeTotalCoverage;
		forestCoverage /= biomeTotalCoverage;
	}

	imageStore(additional0Write, dataPos, info.additionalData[0]);
	imageStore(additional1Write, dataPos, info.additionalData[1]);
	imageStore(additional2Write, dataPos, info.additionalData[2]);
	imageStore(additional3Write, dataPos, info.additionalData[3]);
	imageStore(biome0Write, dataPos, majorBiome.biomeData);
	imageStore(biome1Write, dataPos, minorBiome.biomeData);
	imageStore(biome2Write, dataPos, grass);

	if (WRITE_DEBUG) {
        imageStore(debugOutput0Write, dataPos, vec4(info.temperature.meanCelsius, temperateForest[1], 0.f, 0.f));
	}
}