// Compute shader for the GPU simulation that generates plant information
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;
layout (binding = 1) uniform sampler2DRect waterOffset0Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect plants0Write;
layout (binding = 1, r32f) uniform restrict image2DRect plants1Write;

uniform int water_resolution;
uniform ivec2 water_size;
uniform int plants_resolution;
uniform ivec2 plants_offset;
uniform int land_resolution;
uniform bool useWaterOffset;
uniform bool useTriangleSwitching;
uniform bool trianglesOpposite;
const float WATER_LOWER_LIMIT = 2.0;
const float RIVER_THRESHOLD = 0.02;

vec3 calculateNormal(vec3 p1, vec3 p2, vec3 p3) {
    return normalize(cross(p2 - p1, p3 - p1));
}

vec3 getBarycentricCoordinates(vec2 v0, vec2 v1, vec2 v2, vec2 p) {
    // Compute the vectors from v0 to the other vertices and the point
    vec2 v0v1 = v1 - v0;
    vec2 v0v2 = v2 - v0;
    vec2 v0p = p - v0;

    // Compute the dot products
    float d00 = dot(v0v1, v0v1);
    float d01 = dot(v0v1, v0v2);
    float d11 = dot(v0v2, v0v2);
    float d20 = dot(v0p, v0v1);
    float d21 = dot(v0p, v0v2);

    // Compute the denominator of the barycentric coordinates
    float denom = d00 * d11 - d01 * d01;

    // Compute the barycentric coordinates
    float v = (d11 * d20 - d01 * d21) / denom;
    float w = (d00 * d21 - d01 * d20) / denom;
    float u = 1.0 - v - w;

    return vec3(u, v, w);
}

bool isLeft(vec2 linePoint1, vec2 linePoint2, vec2 point) {
	return ((linePoint2.x - linePoint1.x) * (point.y - linePoint1.y)) < (((point.x - linePoint1.x) * (linePoint2.y - linePoint1.y)));
}

bool insideSquare(vec2 points[4], vec2 point) {
	bool inside = false;
	int point1Index = 0, point2Index = 3;

	if (points[0] == point) {
		return true;
	}

	for (int i=0; i<4; i++) {
		if (((points[point1Index].y > point.y) != (points[point2Index].y > point.y)) &&
			(point.x < (points[point2Index].x - points[point1Index].x) * (point.y - points[point1Index].y) / (points[point2Index].y - points[point1Index].y) + points[point1Index].x)) {
			inside = !inside;
		}
		point2Index = point1Index;
		point1Index++;
	}
	return inside;
}

const ivec2 squaresToCheck[25] = ivec2[25](
	ivec2(0, 0), ivec2(0, -1), ivec2(0, 1), ivec2(-1, 0), ivec2(1, 0), 
	ivec2(-1, -1), ivec2(1, -1), ivec2(-1, 1), ivec2(1, 1), ivec2(-2, 0), 
	ivec2(-2, -1), ivec2(-2, 1),  ivec2(-2, -2),  ivec2(-2, 2), ivec2(-1, -2), 
	ivec2(0, -2), ivec2(1, -2), ivec2(2, -2), ivec2(2, -1), ivec2(2, 0), 
	ivec2(2, 1), ivec2(2, 2), ivec2(1, 2), ivec2(0, 2), ivec2(-1, 2)
);

ivec2 findSquare(vec2 textureCoordinates, int scaling) {
	ivec2 topLeftSquareCoordinates = (ivec2(int(textureCoordinates.x), int(textureCoordinates.y)) / scaling) + (plants_offset / scaling);
	ivec2 topLeftOffset = ivec2(0, 0);
	vec2 waterCoordinates = ((textureCoordinates - vec2(0.5, 0.5)) / float(scaling)) + (plants_offset / scaling);
	bool rightEdge = (topLeftSquareCoordinates.x >= (water_size.x - 1));
	bool bottomEdge = (topLeftSquareCoordinates.y >= (water_size.y - 1));
	if (rightEdge || bottomEdge) {
		return ivec2(rightEdge ? -1 : 0, bottomEdge ? -1 : 0);
	}

	for (int i=0; i<25; i++) {
		topLeftOffset = squaresToCheck[i];

		ivec2 _topLeftCoordinates = topLeftSquareCoordinates + topLeftOffset;
		if (_topLeftCoordinates.x < 0 || _topLeftCoordinates.y < 0 || 
			_topLeftCoordinates.x >= water_size.x || _topLeftCoordinates.y >= water_size.y) continue;
		vec2 topLeftTexel = _topLeftCoordinates + vec2(0.5, 0.5);
		const float delta = 0.00025;

		vec2 currentLocalCoordinates = waterCoordinates - _topLeftCoordinates;
		vec2 points[4];
		points[0] = vec2(-delta, -delta) + texture(waterOffset0Read, topLeftTexel).xy;
		points[1] = vec2(1 + delta, -delta) + texture(waterOffset0Read, topLeftTexel + vec2(1, 0)).xy;
		points[2] = vec2(1 + delta, 1 + delta) + texture(waterOffset0Read, topLeftTexel + vec2(1, 1)).xy;
		points[3] = vec2(-delta, 1 + delta) + texture(waterOffset0Read, topLeftTexel + vec2(0, 1)).xy;

		if (insideSquare(points, currentLocalCoordinates)) {
			return topLeftOffset;
		}
	}

	return ivec2(-25, -25); 
}

int scale(int fromResolution, int toResolution) {
	int numerator = 1, denominator = 1;
	if (fromResolution > 0) {
		numerator *= fromResolution;
	} else {
		denominator *= -fromResolution;
	}
	if (toResolution > 0) {
		denominator *= toResolution;
	} else {
		numerator *= -toResolution;
	}
	return numerator / denominator;
}

float scaleFloat(int fromResolution, int toResolution) {
	float numerator = 1, denominator = 1;
	if (fromResolution > 0) {
		numerator *= fromResolution;
	} else {
		denominator *= -fromResolution;
	}
	if (toResolution > 0) {
		denominator *= toResolution;
	} else {
		numerator *= -toResolution;
	}
	return numerator / denominator;
}

void main() {

	int scaling = scale(water_resolution, plants_resolution);
	float height = 0, waterHeight = 0, debug1 = 0, debug2 = 0;
	ivec2 waterOffset = plants_offset / scaling;

	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);
	vec2 plantsTexCoord = vec2(gl_GlobalInvocationID.xy) + vec2(0.5, 0.5);

	ivec2 topLeft1 = ivec2(int(plantsTexCoord.x), int(plantsTexCoord.y));
	topLeft1 = topLeft1 / scaling;
	vec2 topLeftCoords = vec2(topLeft1.x, topLeft1.y);
	topLeftCoords = topLeftCoords + waterOffset;

	// The texCoord for the first pixel will be (0.5, 0.5) because it is placed in the middle of the pixel
	vec2 waterCoordinates = (vec2(dataPos) / float(scaling)) + waterOffset;
	ivec2 squareOffset = findSquare(plantsTexCoord, scaling);

	if (!useWaterOffset) {
		squareOffset = ivec2(0, 0);
	}
	if (squareOffset.x == -25 || squareOffset.y == -25) {
		debug1 = 1;
	}

	topLeftCoords += squareOffset;
	vec2 topLeftTexel = topLeftCoords + vec2(0.5, 0.5);

	vec4 topLeftData = texture(water0Read, topLeftTexel);
	vec4 topRightData = texture(water0Read, topLeftTexel + (vec2(1, 0)));
	vec4 bottomLeftData = texture(water0Read, topLeftTexel + (vec2(0, 1)));
	vec4 bottomRightData = texture(water0Read, topLeftTexel + (vec2(1, 1)));

	vec4 landHeights = vec4(topLeftData.x, topRightData.x, bottomLeftData.x, bottomRightData.x);
	vec4 waterHeights = vec4(topLeftData.w, topRightData.w, bottomLeftData.w, bottomRightData.w);

	vec4 topLeftOffset = texture(waterOffset0Read, topLeftTexel);
	vec4 topRightOffset = texture(waterOffset0Read, topLeftTexel + (vec2(1, 0)));
	vec4 bottomLeftOffset = texture(waterOffset0Read, topLeftTexel + (vec2(0, 1)));
	vec4 bottomRightOffset = texture(waterOffset0Read, topLeftTexel + (vec2(1, 1)));
	if (!useWaterOffset) {
		topLeftOffset.x = topRightOffset.x = bottomLeftOffset.x = bottomRightOffset.x = 0.0;
		topLeftOffset.y = topRightOffset.y = bottomLeftOffset.y = bottomRightOffset.y = 0.0;
	}

	vec3 topLeft = vec3(topLeftCoords.x, 0, topLeftCoords.y) + vec3(topLeftOffset.x, 0, topLeftOffset.y);
	vec3 topRight = vec3(topLeftCoords.x + 1.0, 0, topLeftCoords.y) + vec3(topRightOffset.x, 0, topRightOffset.y);
	vec3 bottomLeft = vec3(topLeftCoords.x, 0, topLeftCoords.y + 1.0) + vec3(bottomLeftOffset.x, 0, bottomLeftOffset.y);
	vec3 bottomRight = vec3(topLeftCoords.x + 1.0, 0, topLeftCoords.y + 1.0) + vec3(bottomRightOffset.x, 0, bottomRightOffset.y);

	bool flipTriangles = false;
	if (useTriangleSwitching) {
		flipTriangles = topLeftOffset.b == 1;
	}
	if (trianglesOpposite) {
		flipTriangles = !flipTriangles;
	}

	const vec4 xCoords = vec4(0, 1, 0, 1), zCoords = vec4(0, 0, 1, 1);
	vec2 v1, v2, v3, point;
	vec4 c1, c2, c3;
	vec3 normal;

	if (flipTriangles) {
		v1 = topLeft.xz;
		c1 = vec4(1, 0, 0, 0);

		if (isLeft(topLeft.xz, bottomRight.xz, waterCoordinates)) {
			v2 = bottomRight.xz;
			v3 = topRight.xz;
			c2 = vec4(0, 0, 0, 1);
			c3 = vec4(0, 1, 0, 0);
			normal = calculateNormal(vec3(v1.x * 16.f, landHeights[0], v1.y * 16.f), vec3(v2.x * 16.f, landHeights[3], v2.y * 16.f), vec3(v3.x * 16.f, landHeights[1], v3.y * 16.f));
		} else {
			v2 = bottomLeft.xz;
			c2 = vec4(0, 0, 1, 0);
			v3 = bottomRight.xz;
			c3 = vec4(0, 0, 0, 1);
			normal = calculateNormal(vec3(v1.x * 16.f, landHeights[0], v1.y * 16.f), vec3(v2.x * 16.f, landHeights[2], v2.y * 16.f), vec3(v3.x * 16.f, landHeights[3], v3.y * 16.f));
		}
	} else {
		v1 = topRight.xz;
		c1 = vec4(0, 1, 0, 0);
		
		if (isLeft(bottomLeft.xz, topRight.xz, waterCoordinates)) {
			v2 = topLeft.xz;
			c2 = vec4(1, 0, 0, 0);
			v3 = bottomLeft.xz;
			c3 = vec4(0, 0, 1, 0);
			normal = calculateNormal(vec3(v1.x * 16.f, landHeights[1], v1.y * 16.f), vec3(v2.x * 16.f, landHeights[0], v2.y * 16.f), vec3(v3.x * 16.f, landHeights[2], v3.y * 16.f));
		} else {
			v2 = bottomLeft.xz;
			v3 = bottomRight.xz;
			c2 = vec4(0, 0, 1, 0);
			c3 = vec4(0, 0, 0, 1);
			normal = calculateNormal(vec3(v1.x * 16.f, landHeights[1], v1.y * 16.f), vec3(v2.x * 16.f, landHeights[2], v2.y * 16.f), vec3(v3.x * 16.f, landHeights[3], v3.y * 16.f));
		}
	}

	vec3 barycentricCoordinates = getBarycentricCoordinates(v1, v2, v3, waterCoordinates);
	vec4 amounts = (barycentricCoordinates[0] * c1) + (barycentricCoordinates[1] * c2) + (barycentricCoordinates[2] * c3);
	height = dot(landHeights * amounts, vec4(1));
	waterHeight = dot(waterHeights * amounts, vec4(1));
	float finalXCoord = dot(xCoords * amounts, vec4(1));
	float finalZCoord = dot(zCoords * amounts, vec4(1));

	vec2 finalWaterCoordinates = topLeftTexel + 
		(vec2(1, 0) * amounts[1]) + 
		(vec2(0, 1) * amounts[2]) + 
		(vec2(1, 1) * amounts[3]); 

	if (debug1 == 1) {
		height = -1;
	}

	imageStore(plants0Write, dataPos, vec4(height, waterHeight, finalWaterCoordinates.x, finalWaterCoordinates.y));
	imageStore(plants1Write, dataPos, vec4(normal.y, 0, 0, 0));
}