// Compute shader for the GPU simulation that generates data for specific trees
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;
layout (binding = 1) uniform sampler2DRect plants0Read;
layout (binding = 2) uniform sampler2DRect plants1Read;
layout (binding = 3) uniform sampler2DRect biomes0Read;
layout (binding = 4) uniform sampler2DRect biomes1Read;
layout (binding = 5) uniform sampler2D biomePalette;

layout (binding = 0, rgba32f) uniform restrict image2DRect trees0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect trees1Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect debugOutput0Write;

#include "common/constants.glsl"
#include "common/data/biomes.glsl"
#include "common/noise.glsl"
#include "common/resolutions.glsl"

uniform ivec2 water_offset;
uniform ivec2 water_size;
uniform int water_resolution;
uniform ivec2 plants_offset;
uniform ivec2 plants_size;
uniform int plants_resolution;
uniform ivec2 trees_offset;
uniform ivec2 trees_size;
uniform int trees_resolution;
uniform bool WRITE_DEBUG;
uniform float PLANT_DENSITY_MULTIPLIER;
uniform float GLOBAL_OBJECT_SCALE;

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	vec2 worldCoodinates = convertCoordinates(dataPos, trees_resolution, 1, trees_offset, vec2(0, 0));
	ivec2 plantsPosition = ivec2(convertCoordinates(dataPos, trees_resolution, plants_resolution, trees_offset, plants_offset));
	vec4 plants0Data = texelFetch(plants0Read, plantsPosition);
	vec4 plants1Data = texelFetch(plants1Read, plantsPosition);

	vec2 waterPosition = convertCoordinates(dataPos, trees_resolution, water_resolution, trees_offset, water_offset) + vec2(0.5, 0.5);
	vec2 waterTexel = plants0Data.zw - vec2(0.5, 0.5);
	vec2 waterFractPos = fract(waterTexel);
	ivec2 waterDataPos = ivec2(waterTexel);
	vec4 noise = texture(noiseTexture, dataPos / noiseTextureSize);
	if (noise[0] > waterFractPos.x) {
		if (noise[1] < waterFractPos.y) {
			waterDataPos += ivec2(0, 1);
		}
	} else {
		waterDataPos += ivec2(1, 0);
		if (noise[1] < waterFractPos.y) {
			waterDataPos += ivec2(0, 1);
		}
	}

	vec4 biomesData[2];
	biomesData[0] = texelFetch(biomes0Read, waterDataPos);
	biomesData[1] = texelFetch(biomes1Read, waterDataPos);

	float rotation = rand(vec2(plantsPosition.x * 13, plantsPosition.y * 7)) * M_PI * 2.f;
	int type = int(rand(vec2(plantsPosition.x * 47, plantsPosition.y * 13)) * 3);
	int existsOutput = 0;

	bool exists = false;
	int treeId = 0;
	int biomeId = -1, biomeIndex = 0;
	ivec2 biomeIds = ivec2(int(biomesData[0][0] * 255.f), int(biomesData[1][0] * 255.f));
	vec2 biomeChances = vec2(biomesData[0][1], biomesData[1][1]);
	float treeChance = rand(vec2(plantsPosition.x * 71, plantsPosition.y * 83));
	
	float biomeChance = rand(vec2(plantsPosition.x * (22 + biomeId), plantsPosition.y * 453));
	if (biomeChances[0] > biomeChance) {
		biomeId = biomeIds[0];
	} else {
		if (biomeChances[0] + biomeChances[1] > biomeChance) {
			biomeId = biomeIds[1];
			biomeIndex = 1;
		}
	}

	if (treeChance > PLANT_DENSITY_MULTIPLIER || biomeId == -1) {
		exists = false;
	} else {
		// since the previous if statement chops off the top PLANT_DENSITY_MULTIPLIER portion of the random distrubtion
		treeChance = treeChance / PLANT_DENSITY_MULTIPLIER;

		for (int i=0; i<16; i++) {
			if (biomeDefinitions[biomeId].trees[i].probabilityLimit > treeChance) {
				treeId = biomeDefinitions[biomeId].trees[i].id;
				if (treeId != 0) {
					exists = true;
				}
				break;
			}
		}
	}

	float waterDepth = plants0Data[1];
	if (plants0Data[0] < 0.f) {
		waterDepth = max(-plants0Data[0], waterDepth);
	}

	if (waterPosition.x < 0 || waterPosition.y < 0 || 
		waterPosition.x >= water_size.x - 0.5 || waterPosition.y >= water_size.y - 0.5 || 
		plants1Data[0] < 0.5f || 
		(waterDepth / GLOBAL_OBJECT_SCALE) > plantDefinitions[treeId].maxWaterDepth) {
		exists = false;
	}

	float variation = 0.1f;
	variation = 0.02f;

	vec4 colour = vec4(0);
	if (exists) {
		colour = texture(biomePalette, vec2(biomesData[biomeIndex][2], (0.5 / 16.f) + (biomeId / 16.f)));
	}

	float height = 0.75f + (rand(dataPos.yx)) * 0.5f;

	imageStore(trees0Write, dataPos, vec4(plants0Data.x, exists ? treeId : 0, rotation, height));
	imageStore(trees1Write, dataPos, vec4(colour.r, colour.g, colour.b, colour.a < 0.5f ? 0.f : 1.f));
}