// Fragment shader for the GPU water flow simulation pass that calculates occlusion for land
#version 430

#include "common/data/rocks.glsl"

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect rockProperties0Read;
layout (binding = 1) uniform sampler2DRect rockProperties1Read;
layout (binding = 2) uniform sampler2DRect water0Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect rockProperties0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect rockProperties1Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect debugOutput0Write;

uniform bool WRITE_DEBUG;

#include "common/constants.glsl"
#include "common/noise.glsl"


float illumination(float dist, float relativeHeight) {
	return min(1.0, 1.0 - (atan(relativeHeight, dist) / (M_PI / 2)));
}

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	vec4 currentValues = texelFetch(rockProperties0Read, dataPos);
	vec4 heights = texelFetch(water0Read, dataPos);
	float occludedLightTotal = 0;
	float thisHeight = max(0.f, heights[2]);
	float elapsedSinceChange = currentValues.z;
	vec4 neighbourHeights = vec4(0);

	float dirtDepth = currentValues[1];
	float rockType = currentValues[2];
	float rockDepth = currentValues[3];
	vec4 debugOutput;

	RockLayers rockLayers = unpackRockLayers(texelFetch(rockProperties1Read, dataPos));
	if (rockDepth <= 0.f) {
		if (rockLayers.layers[0].type == 0) {
			dirtDepth += rockLayers.layers[0].depth;
		} else {
			rockType = rockLayers.layers[0].type;
			rockDepth = rockLayers.layers[0].depth;
		}
		pullUpRockLayers(rockLayers);
	}

	debugOutput = vec4(float(rockLayers.layers[0].type), rockLayers.layers[0].depth, float(rockLayers.layers[1].type), rockLayers.layers[1].depth);

	vec4 output1 = vec4(0.f, dirtDepth, rockType, rockDepth);
	imageStore(rockProperties0Write, dataPos, output1);
	imageStore(rockProperties1Write, dataPos, packRockLayers(rockLayers));

	if (WRITE_DEBUG) {
		imageStore(debugOutput0Write, dataPos, debugOutput);
	}
}