// Fragment shader for the second step of the GPU water flow simulation
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;
layout (binding = 1) uniform sampler2DRect water1Read;
layout (binding = 2) uniform sampler2DRect water2Read;
layout (binding = 3) uniform sampler2DRect water3Read;
layout (binding = 4) uniform sampler2DRect lava0Read;
layout (binding = 5) uniform sampler2DRect rockProperties0Read;
layout (binding = 6) uniform sampler2DRect temperature0Read;
layout (binding = 7) uniform sampler2DRect ice0Read;
layout (binding = 8) uniform sampler2DRect flood0Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect water0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect water2Write;
layout (binding = 2, rgba32f) uniform restrict image2DRect water3Write;
layout (binding = 3, rgba32f) uniform restrict image2DRect rockProperties0Write;
layout (binding = 4, rgba32f) uniform restrict image2DRect debugOutput0Write;
layout (binding = 5, rgba32f) uniform restrict image2DRect debugOutput1Write;

uniform int water_resolution;
uniform float THERMAL_EROSION_CUTOFF;
uniform float THERMAL_EROSION_SPEED;
uniform float TIMESTEP;
uniform float SEDIMENT_CAPACITY;
uniform float DEPOSIT_AMOUNT;
uniform float EROSION_AMOUNT;
uniform float WATER_EROSION_MULTIPLIER;
uniform float WATER_DEPOSIT_MULTIPLIER;
uniform float SIMULATION_SPEED;
uniform float RAIN_STRENGTH;
uniform float FLOW_PERCENT;
uniform float SLOPE_OFFSET;
uniform float EVAPORATION;
uniform float WATER_LOWER_LIMIT;
uniform float RIVER_THRESHOLD;
uniform float OCEAN_DEPOSIT_MULTIPLIER;
uniform float RAINFALL_NORMALISED_TO_SIM_CONVERSION;
uniform float FLOOD_DEPOSITION_MULTIPLIER;
uniform float FLOOD_DEPOSIT_AMOUNT;
uniform float FLOOD_HEIGHT_MULTIPLIER;
uniform float NON_RIVER_EROSION_MULTIPLIER;
uniform float FLOOD_IDEAL_SLOPE;
uniform float FLOOD_SLOPE_EXPONENT;
uniform float FLOOD_SLOPE_DEPOSIT_LIMIT;
uniform float OCEAN_DISCONNECTED_FLOOD_DEPOSIT_RATIO;
uniform float DIRT_EROSION_SLOPE_MODIFIER;
uniform float GROUNDWATER_DIFFUSE_RATE;
uniform float GROUNDWATER_WATER_SOAK_RATE;
uniform float GROUNDWATER_WATER_SOAK_REMOVED;
uniform float GROUNDWATER_WATER_SOAK_LIMIT;
uniform float GROUNDWATER_DRY_RATE;
uniform int step_count;
uniform bool WRITE_DEBUG;
uniform bool DISPLAY_FLOOD;
uniform float GLOBAL_RAINFALL_MODIFIER;

#include "common/erosion.glsl"
#include "common/noise.glsl"
#include "common/data/water.glsl"
#include "common/util.glsl"
#include "common/temperature.glsl"
#include "common/interpolations.glsl"

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	vec4 debugOutput1 = vec4(0), debugOutput2 = vec4(0);

	WaterData data = getWaterData(dataPos, true, true, true);
	WaterData neighbours[8];
	neighbours[0] = getWaterData(dataPos + ivec2(0, -1), true, true, true);
	neighbours[1] = getWaterData(dataPos + ivec2(1, 0), true, true, true);
	neighbours[2] = getWaterData(dataPos + ivec2(0, 1), true, true, true);
	neighbours[3] = getWaterData(dataPos + ivec2(-1, 0), true, true, true);
	neighbours[4] = getWaterData(dataPos + ivec2(-1, -1), false, false, false);
	neighbours[5] = getWaterData(dataPos + ivec2(1, -1), false, false, false);
	neighbours[6] = getWaterData(dataPos + ivec2(-1, 1), false, false, false);
	neighbours[7] = getWaterData(dataPos + ivec2(1, 1), false, false, false);

	vec4 floodNeighbours[4];
	floodNeighbours[0] = texelFetch(flood0Read, dataPos + ivec2(0, -1));
	floodNeighbours[1] = texelFetch(flood0Read, dataPos + ivec2(1, 0));
	floodNeighbours[2] = texelFetch(flood0Read, dataPos + ivec2(0, 1));
	floodNeighbours[3] = texelFetch(flood0Read, dataPos + ivec2(-1, 0));
	vec4 floodFlows = texelFetch(flood0Read, dataPos);

	vec4 lava = texelFetch(lava0Read, dataPos);
	vec4 rockProperties = texelFetch(rockProperties0Read, dataPos);
	vec4 ice = texelFetch(ice0Read, dataPos);
	vec4 temperature = texelFetch(temperature0Read, dataPos);

	float depositMultiplier = SIMULATION_SPEED * WATER_DEPOSIT_MULTIPLIER;
	float erosionMultiplier = SIMULATION_SPEED * WATER_EROSION_MULTIPLIER;
	if (data.waterHeight < RIVER_THRESHOLD) {
		erosionMultiplier *= NON_RIVER_EROSION_MULTIPLIER;
	}
	ErosionParams erosionParams = getErosionParams(rockProperties);
	ErosionParams rockErosionParams = getRockErosionParams(rockProperties);
	float dirtErosionAmount = EROSION_AMOUNT * erosionParams.regularErosionSpeed;
	float rockErosionAmount = EROSION_AMOUNT * rockErosionParams.regularErosionSpeed;
	if (rockProperties[1] > 1.f) {
		rockErosionAmount = 0.f;
	}
	float depositAmount = DEPOSIT_AMOUNT;
	// depositAmount /= (clamp((data.waterHeight - 2.f), 1.f, 20.f));
	
	float previousSediment = data.sediment;
	float previousFloodSediment = data.floodSediment;
	float lowestNeighbourWater = 999999.f;
	float waterAdd = 0.f;

	for (int i=0; i<4; i++) {
		float sedimentPercentTransfer = 0.f;
		if (data.waterHeight != 0.f) {
			sedimentPercentTransfer = min(1.0, (data.flows[i] * TIMESTEP) / data.waterHeight);
		}
		data.sediment -= (previousSediment * sedimentPercentTransfer);
		waterAdd -= data.flows[i];

		float floodSedimentPercentTransfer = 0.f;
		if (data.floodHeight != 0.f) {
			floodSedimentPercentTransfer = min(1.0, floodFlows[i] / data.floodHeight);
		}
		data.floodSediment -= (previousFloodSediment * floodSedimentPercentTransfer);
	}

	float ourWater = max(data.waterHeight, 0.0000001f);
	float sedimentToAdd = 0.f, floodSedimentToAdd = 0.f;
	int primaryInflowDirection = 0;
	float largestInflow = 0.f;
	float floodAdd = 0.f;

	for (int i=0; i<4; i++) {
		int opposite = oppositeDirection(i);
		float flow = neighbours[i].flows[opposite];
		if (flow > largestInflow) {
			largestInflow = flow;
			primaryInflowDirection = i;
		}
		float floodFlow = floodNeighbours[i][opposite];
		floodAdd -= floodFlows[i];
		floodAdd += floodFlow;

		waterAdd += flow;
		if (neighbours[i].waterHeight != 0.f) {
			float sedimentPercentTransfer = min(1.f, ((flow * TIMESTEP) / neighbours[i].waterHeight));
			sedimentToAdd += sedimentPercentTransfer * neighbours[i].sediment;			
		}
		if (neighbours[i].floodHeight != 0.f) {
			float sedimentPercentTransfer = min(1.f, floodFlow / neighbours[i].floodHeight);
			floodSedimentToAdd += sedimentPercentTransfer * neighbours[i].floodHeight;
		}

		float neighbourWaterHeight = max(0.f, neighbours[i].landHeight + neighbours[i].waterHeight);
		if ((neighbours[i].waterHeight > RIVER_THRESHOLD || neighbours[i].landHeight < 0.f) && neighbourWaterHeight < lowestNeighbourWater) {
			lowestNeighbourWater = neighbourWaterHeight;
		}
	}

	for (int i=4; i<8; i++) {
		float neighbourWaterHeight = max(0.f, neighbours[i].landHeight + neighbours[i].waterHeight);
		if ((neighbours[i].waterHeight > RIVER_THRESHOLD || neighbours[i].landHeight < 0.f) && neighbourWaterHeight < lowestNeighbourWater) {
			lowestNeighbourWater = neighbourWaterHeight;
		}
	}

	// if (data.waterHeight > RIVER_THRESHOLD) {
	// 	if (data.primaryFlowDirection != oppositeDirection(primaryInflowDirection)) {
	// 		data.outsideBank = oppositeDirection(data.primaryFlowDirection);
	// 	} else {
	// 		data.outsideBank = neighbours[data.primaryFlowDirection].outsideBank;
	// 	}
	// }

	if ((data.landHeight + data.waterHeight) < (neighbours[data.primaryFlowDirection].waterHeight + neighbours[data.primaryFlowDirection].landHeight)) {
		data.connectsOcean -= 1.f;
	} else {
		data.connectsOcean = neighbours[data.primaryFlowDirection].connectsOcean;
	}
	if (neighbours[data.primaryFlowDirection].landHeight < 0) {
		data.connectsOcean = 120.f;
	}
	data.connectsOcean = max(data.connectsOcean, 0.f);

	float heightForwards = neighbours[data.primaryFlowDirection].landHeight;
	float differenceForwards = data.landHeight - heightForwards;
	// Erosion multiplier due to local slope- the higher the slope, the more erosion and the less deposition
	float slopeFactor = sin(atan(differenceForwards, float(water_resolution))) + SLOPE_OFFSET;
	slopeFactor = max(0.f, slopeFactor);
	// Oceans only deposit sediment, don't erode
	float oceanFactor = clamp(1.f - (-data.landHeight / 8.f), 0.f, 1.f);
	float erosionFactor = oceanFactor * slopeFactor;

	float deposition = 0.f;

	float temperatureCelsius = normalisedToCelsius(temperature[0]);
	if (temperatureCelsius > 0.f) {
		data.waterHeight += data.rainfall * RAIN_STRENGTH * GLOBAL_RAINFALL_MODIFIER;
		data.floodHeight += data.rainfall * FLOOD_HEIGHT_MULTIPLIER * GLOBAL_RAINFALL_MODIFIER;
	}

	data.waterHeight += ice[2] * GLOBAL_RAINFALL_MODIFIER;
	if (data.floodHeight < 64.f) {
		data.floodHeight += ice[2] * FLOOD_HEIGHT_MULTIPLIER * GLOBAL_RAINFALL_MODIFIER;
	}


	if (ice[3] < 0.f) {
		sedimentToAdd += -ice[3] * 1.f;
		ice[3] = 0.f;
	}

	data.sediment = data.sediment + sedimentToAdd;
	data.floodSediment = data.floodSediment + floodSedimentToAdd;

	data.waterHeight += (waterAdd * TIMESTEP);
	data.floodHeight += floodAdd;

	if (data.waterHeight > RIVER_THRESHOLD) {
		data.waterHeight -= (EVAPORATION * RAINFALL_NORMALISED_TO_SIM_CONVERSION * GLOBAL_RAINFALL_MODIFIER);
		data.floodHeight -= (EVAPORATION * RAINFALL_NORMALISED_TO_SIM_CONVERSION * FLOOD_HEIGHT_MULTIPLIER * GLOBAL_RAINFALL_MODIFIER);
	}
	data.waterHeight = max(0.0, data.waterHeight);
	data.floodHeight = max(0.0, data.floodHeight);

	float flowRatio = 1.f;
	float velocity = 0.f;
	vec2 velocityComponents = vec2(0,0);	
	velocityComponents[0] -= (data.flows[0] * flowRatio);
	velocityComponents[0] += (data.flows[2] * flowRatio);
	velocityComponents[1] -= (data.flows[1] * flowRatio);
	velocityComponents[1] += (data.flows[3] * flowRatio);
	velocity = length(velocityComponents);
	velocity = velocity * (1.f + exponentialOut(min(data.waterHeight / 2.f, 1.f)));

	float sedimentCapacity = velocity * SEDIMENT_CAPACITY * erosionFactor;

	/* In the ocean, water should spread sediment all around, but not pool too much */
	if (data.landHeight < 0) {
		sedimentCapacity = 0.f;
		depositMultiplier *= OCEAN_DEPOSIT_MULTIPLIER;
		data.sediment = mix(data.sediment, 0.f, 0.01f);
		data.waterHeight = mix(data.waterHeight, 0.f, 0.01f);
		data.floodHeight = mix(data.floodHeight, 0.f, 0.01f);
		data.floodSediment = mix(data.floodSediment, 0.f, 0.01f);
	}

	float regularDifference = 0.f;
	float regularErosion = 0.f, regularDeposition = 0.f;
	float floodDifference = 0.f;
	float iceDifference = ice[3] * SIMULATION_SPEED;
	float dirtErosion = iceDifference, rockErosion = 0.f;
	if (rockErosionAmount != 0.f) {
		rockErosion = ice[3] * SIMULATION_SPEED;
	}
	float sedimentDifferenceErosion = sedimentCapacity * clamp(cubicIn(sedimentCapacity / data.sediment), 0.f, 1.f);

	float sedimentDifferenceDeposition = sedimentCapacity - data.sediment;
	if (sedimentDifferenceDeposition < 0.f) {
		float sedimentToDeposit = depositAmount * -sedimentDifferenceDeposition * TIMESTEP;
		sedimentToDeposit = min(sedimentToDeposit, data.sediment);
		data.sediment -= sedimentToDeposit;
		regularDeposition = (sedimentToDeposit * depositMultiplier * 1.f);
		deposition += regularDeposition;
		regularDifference += regularDeposition;
	}

	float dirtSedimentToErode = dirtErosionAmount * sedimentDifferenceErosion * TIMESTEP * clamp(rockProperties[1] / DIRT_EROSION_SLOPE_MODIFIER, 0.f, 1.f);
	dirtSedimentToErode = min(dirtSedimentToErode, 0.005 * WATER_EROSION_MULTIPLIER);
	float rockSedimentToErode = rockErosionAmount * sedimentDifferenceErosion * TIMESTEP;
	rockSedimentToErode = min(rockSedimentToErode, 0.005 * WATER_EROSION_MULTIPLIER);

	float floodSlope = pow(clamp(differenceForwards / FLOOD_IDEAL_SLOPE, 0.f, 1000.f), FLOOD_SLOPE_EXPONENT);
	if (differenceForwards > FLOOD_SLOPE_DEPOSIT_LIMIT) {
		floodSlope = 100000.f;
	}
	float floodSedimentCapacity = (data.floodHeight * 0.1f) * SEDIMENT_CAPACITY * floodSlope * 0.01f;
	float floodSedimentDifference = floodSedimentCapacity - data.floodSediment;

	if (ice[0] > 0.f && data.floodHeight < 8.f) {
		floodSedimentDifference = 0.f;
	}

	if (data.landHeight > 0.f) {
		if (floodSedimentDifference < 0.f) {
			float sedimentToDeposit = FLOOD_DEPOSIT_AMOUNT * -floodSedimentDifference;
			if (data.connectsOcean == 0.f) {
				sedimentToDeposit *= OCEAN_DISCONNECTED_FLOOD_DEPOSIT_RATIO;
			}
			data.floodSediment -= sedimentToDeposit;
			data.floodSediment = max(0.f, data.floodSediment);
			floodDifference = (sedimentToDeposit * FLOOD_DEPOSITION_MULTIPLIER * SIMULATION_SPEED);
			deposition += floodDifference;
		} else {
			float sedimentAdd = (dirtSedimentToErode + rockSedimentToErode);
			data.floodSediment += sedimentAdd;
		}
	}

	regularErosion = -(dirtSedimentToErode * erosionMultiplier);
	dirtErosion += -regularErosion;
	regularDifference += regularErosion;
	rockErosion += (rockSedimentToErode * erosionMultiplier);

	data.sediment += (dirtSedimentToErode + rockSedimentToErode);
	data.sediment = clamp(data.sediment, 0.f, SEDIMENT_CAPACITY * 16.f);

	if (deposition > 0.f) {
		rockProperties[1] += deposition;
		data.landHeight += deposition;
	}
	if (dirtErosion > 0.f) {
		dirtErosion = min(dirtErosion, rockProperties[1]);
		rockProperties[1] -= dirtErosion;
		data.landHeight -= dirtErosion;
	}
	if (rockErosion > 0.f) {
		rockErosion = min(rockErosion, rockProperties[3]);
		rockProperties[3] -= rockErosion;
		data.landHeight -= rockErosion;
	}

	if (WRITE_DEBUG) {
		debugOutput1[0] = data.landHeight;
		debugOutput1[1] = data.waterHeight;
		debugOutput1[2] = regularDifference * 1000.f;
		// debugOutput1[3] = floodDifference * 1000.f;

		// debugOutput2[0] = iceDifference * 1000.f;
		// debugOutput2[1] = regularErosion * 1000.f;

		debugOutput1[3] = data.sediment;
		debugOutput2[0] = dirtErosion * 1000.f;
		debugOutput2[1] = rockErosion * 1000.f;

		debugOutput2[2] = regularDeposition * 1000.f;
		debugOutput2[3] = rockProperties[1];
	}

	float waterEvaporatedDueToLava = min(data.waterHeight * 0.25f, lava[0]);
	if (waterEvaporatedDueToLava > 0) {
		float waterEvaporatedDueToLavaPercent = waterEvaporatedDueToLava / data.waterHeight;
		data.sediment -= (data.sediment * waterEvaporatedDueToLavaPercent);
	}
 	data.waterHeight = max(data.waterHeight - waterEvaporatedDueToLava, 0.f);

 	if (lava[0] > 0.f) {
 		data.floodHeight = 0.f;
 		data.floodSediment = 0.f;
 	}

 	float averageNeighbourGroundwater = 0.f;
 	for (int i=0; i<4; i++) {
 		averageNeighbourGroundwater += neighbours[i].groundWater;
 	}
 	averageNeighbourGroundwater /= 4.f;
 	data.groundWater = mix(data.groundWater, averageNeighbourGroundwater, GROUNDWATER_DIFFUSE_RATE);

 	float rainfallNormalised = data.rainfall / RAINFALL_NORMALISED_TO_SIM_CONVERSION;
 	data.groundWater = mix(data.groundWater, rainfallNormalised, GROUNDWATER_DRY_RATE);

 	float waterToSoak = (data.waterHeight - RIVER_THRESHOLD) > 0.f ? min(rainfallNormalised + GROUNDWATER_WATER_SOAK_LIMIT, 1.f) : 0.f;
 	if (waterToSoak > data.groundWater) {
 		data.groundWater = mix(data.groundWater, waterToSoak, GROUNDWATER_WATER_SOAK_RATE);
 		data.waterHeight = max(data.waterHeight - ((waterToSoak - data.groundWater) * GROUNDWATER_WATER_SOAK_REMOVED), 0.f);
 	}

 	if (data.landHeight < 0.f) {
 		data.waterDisplayHeight = -data.landHeight;
 	} else {
 		float _waterHeight = DISPLAY_FLOOD ? data.floodHeight : data.waterHeight;
	 	if (_waterHeight < RIVER_THRESHOLD) {
	 		data.waterDisplayHeight = min(lowestNeighbourWater - data.landHeight - WATER_LOWER_LIMIT, -WATER_LOWER_LIMIT);
	 	} else {
	 		data.waterDisplayHeight = _waterHeight < 1.f ? exponentialOut(_waterHeight) : _waterHeight;
	 	}
 	}

	vec4 output1, output2, output3, output4;
	outputWaterData(data, output1, output2, output3, output4);
	imageStore(water0Write, dataPos, output1);
	// imageStore(water1Write, dataPos, output2);
	imageStore(water2Write, dataPos, output3);
	imageStore(water3Write, dataPos, output4);
	// imageStore(ice0Write, dataPos, ice);
	imageStore(rockProperties0Write, dataPos, rockProperties);

	if (WRITE_DEBUG) {
		imageStore(debugOutput0Write, dataPos, debugOutput1);
		imageStore(debugOutput1Write, dataPos, debugOutput2);
	}
}