// Fragment shader for the third step of the GPU water flow simulation
#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect water0Read;
layout (binding = 1) uniform sampler2DRect water2Read;
layout (binding = 2) uniform sampler2DRect lava0Read;
layout (binding = 3) uniform sampler2DRect rockProperties0Read;
layout (binding = 4) uniform sampler2DRect ice0Read;
layout (binding = 5) uniform sampler2DRect ice1Read;

/* These are not actually read, but the common code includes references to them so must be defined */
layout (binding = 6) uniform sampler2DRect water1Read;
layout (binding = 7) uniform sampler2DRect water3Read;

layout (binding = 0, rgba32f) uniform restrict image2DRect water0Write;
layout (binding = 1, rgba32f) uniform restrict image2DRect rockProperties0Write;

uniform ivec2 water_size;
uniform float THERMAL_EROSION_SPEED;
uniform float SIMULATION_SPEED;
uniform float TIMESTEP;
uniform float WAVE_EROSION;
uniform float RIVER_THRESHOLD;
uniform int step_count;
uniform float GLOBAL_RAINFALL_MODIFIER;

#include "common/erosion.glsl"
#include "common/noise.glsl"
#include "common/data/water.glsl"
#include "common/util.glsl"

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	WaterData data = getWaterData(dataPos, false, true, false);
	vec4 lava = texelFetch(lava0Read, dataPos);
	vec4 rockProperties = texelFetch(rockProperties0Read, dataPos);
	vec4 ice = texelFetch(ice0Read, dataPos);
	vec4 displayIce = texelFetch(ice1Read, dataPos);

	float dirtDelta = 0.f;
	float rockDelta = 0.f;
	float convertRockToDirt = 0.f;
	ErosionParams erosionParams = getErosionParams(rockProperties);
	ErosionParams rockErosionParams = getRockErosionParams(rockProperties);

	ivec2 offsets[4];
	offsets[0] = ivec2(0, -1);
	offsets[1] = ivec2(1, 0);
	offsets[2] = ivec2(0, 1);
	offsets[3] = ivec2(-1, 0);

	if (lava[0] == 0.f) {
		for (int i=0; i<4; i++) {
			ivec2 newCoords = dataPos + offsets[i];
			if (newCoords.x >= 0.0 && newCoords.x < water_size.x && newCoords.y >= 0.0 && newCoords.y < water_size.y) {
				WaterData other = getWaterData(newCoords, false, true, false);
				vec4 otherRockProperties = texelFetch(rockProperties0Read, newCoords);
				ErosionParams otherErosionParams = getErosionParams(otherRockProperties);
				ErosionParams otherRockErosionParams = getRockErosionParams(otherRockProperties);

				vec4 otherLava = texelFetch(lava0Read, newCoords);
				if (otherLava[0] != 0.f) {
					continue;
				}

				float diff = data.landHeight - other.landHeight;
				/* Cases where this cell is higher than its neighbours */
				if (rockProperties[1] > 0.f && diff > erosionParams.thermalErosionCutoff) {
					float available = min(rockProperties[1], diff - erosionParams.thermalErosionCutoff);
					dirtDelta -= available * erosionParams.thermalErosionSpeed * (SIMULATION_SPEED / 20.f);
				}
				/* Current config values mean thermal erosion isn't used for rocks, but it could be in the future */
				if (rockProperties[1] < 1.f && diff > rockErosionParams.thermalErosionCutoff) {
					float available = min(rockProperties[3], diff - rockErosionParams.thermalErosionCutoff);
					rockDelta -= available * rockErosionParams.thermalErosionSpeed * (SIMULATION_SPEED / 20.f);
				}

				vec2 randSeed = vec2((dataPos.x * 7) + (step_count % 6427), (dataPos.y * 23) + (step_count % 8513));
				float _diff = diff - rockErosionParams.rockErosionCutoff;
				float mult = (diff / 256.f);
				mult *= clamp((4.f - ice[0]) / 4.f, 0.f, 1.f);

				if (
						rockProperties[1] < 1.f && 
						_diff > 0 && 
						rand(randSeed) < (rockErosionParams.rockErosionSpeed * SIMULATION_SPEED * mult)
					) {
					convertRockToDirt = rand(randSeed + vec2(1, 1)) * _diff;
				}				

				/* Cases where this cell is lower than its neighbours */
				if (otherRockProperties[1] > 0.f && diff < -otherErosionParams.thermalErosionCutoff) {
					float available = -min(otherRockProperties[1], -(diff + otherErosionParams.thermalErosionCutoff));
					dirtDelta -= available * otherErosionParams.thermalErosionSpeed * (SIMULATION_SPEED / 20.f);
				}
				/* Current config values mean thermal erosion isn't used for rocks, but it could be in the future */
				if (otherRockProperties[1] < 1.f && diff < -otherRockErosionParams.thermalErosionCutoff) {
					float available = -min(otherRockProperties[3], -(diff + otherRockErosionParams.thermalErosionCutoff));
					rockDelta -= available * otherRockErosionParams.thermalErosionSpeed * (SIMULATION_SPEED / 20.f);
				}
			}
		}

		if (rockProperties[1] < 128.f && ice[0] < 1.f) {
			convertRockToDirt += rockErosionParams.chemicalErosionSpeed * SIMULATION_SPEED * data.rainfall * GLOBAL_RAINFALL_MODIFIER;
		}
	}

	convertRockToDirt = min(convertRockToDirt, rockProperties[3]);
	if (data.landHeight < 8.f) {
		data.landHeight -= clamp(1.f - (abs(data.landHeight) / 16.f), 0.f, 1.f) * WAVE_EROSION * SIMULATION_SPEED;
	}

	rockProperties[1] += dirtDelta;
	if (rockDelta > 0) {
		rockProperties[1] += rockDelta;
	} else {
		rockProperties[3] += rockDelta;
	}

	if (convertRockToDirt > 0.f) {
		rockProperties[1] += convertRockToDirt;
		rockProperties[3] -= convertRockToDirt;
	}

	data.landHeight = clamp(data.landHeight + (dirtDelta + rockDelta), -2048, 4080);
	data.waterHeight = clamp(data.waterHeight, 0, 1024);
 	if (data.landHeight < 0.f) {
 		data.waterDisplayHeight = -data.landHeight;
 	}

	data.totalHeight = calculateTotalHeight(data, lava, displayIce);

	vec4 output1, output2, output3, output4;
	outputWaterData(data, output1, output2, output3, output4);
	imageStore(water0Write, dataPos, output1);
	imageStore(rockProperties0Write, dataPos, rockProperties);
}