#version 430

layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

layout (binding = 0) uniform sampler2DRect rainfall0Read;
layout (binding = 1) uniform sampler2DRect rainfall1Read;
layout (binding = 2) uniform sampler2DRect rainfall2Read;

layout (binding = 0, rgba16) uniform restrict image2DRect rainfall0Write;
layout (binding = 1, rgba16) uniform restrict image2DRect rainfall1Write;
layout (binding = 2, rg16) uniform restrict image2DRect rainfall2Write;

uniform int attributeModes[8];
uniform float attributeAmounts[8];
uniform float attributeVariances[8];
uniform int randomOffsets[8];
uniform int copyFroms[8];
uniform int attributeMappings[8];
uniform ivec2 rainfall_size;
uniform int resolution;

#include "common/constants.glsl"
#include "common/noise.glsl"
#include "common/interpolations.glsl"

void main() {
	ivec2 dataPos = ivec2(gl_GlobalInvocationID.xy);

	vec4 rainfallData0 = texelFetch(rainfall0Read, dataPos);
	vec4 rainfallData1 = texelFetch(rainfall1Read, dataPos);
	vec4 rainfallData2 = texelFetch(rainfall2Read, dataPos);

	for (int i=0; i<8; i++) {
		int mode = attributeModes[i];
		float amount;

		// NOOP- don't change
		if (mode == 0) {
			if (i < 4) {
				amount = rainfallData0[i];
			} else {
				amount = rainfallData1[i - 4];
			}
		}
		// COPY from another attribute
		if (mode == 1) {
			int attributeToCopyFrom = copyFroms[i];
			if (attributeToCopyFrom < 4) {
				amount = rainfallData0[attributeToCopyFrom];
			} else {
				amount = rainfallData1[attributeToCopyFrom - 4];
			}
		}
		// SET to a new, randomly calculated value
		if (mode == 2) {
			float randomness = (1.f + cnoise((dataPos + vec2(randomOffsets[i], randomOffsets[i])) / 128.f)) / 2.3f;
			amount = clamp(attributeAmounts[i] * mix(1.f, randomness, attributeVariances[i]), 0.f, 1.f);
		}
		// SET to linear gradient
		if (mode == 3) {
			amount = clamp(cubicIn(1.f - (float(dataPos.y) / rainfall_size.y)), 0.f, 1.f);
		}

		if (mode == 4) {
			amount = 0.25f;
		}

		if (i < 4) {
			rainfallData0[i] = amount;
		} else {
			rainfallData1[i - 4] = amount;
		}
	}

	float total = 0.f;
	for (int i=0; i<8; i++) {
		int mapping = attributeMappings[i];
		if (mapping < 4) {
			total += rainfallData0[mapping];
		} else {
			total += rainfallData1[mapping - 4];
		}
	}

	imageStore(rainfall0Write, dataPos, rainfallData0);
	imageStore(rainfall1Write, dataPos, rainfallData1);
	imageStore(rainfall2Write, dataPos, vec4(total / 8.f, 0.f, 0.f, 0.f));
}